#ifndef entity_h
#define entity_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi.h"

#include "entitybase.h"
#include "enums.h"
#include "terrain.h"

//############################################################################
// Define ####################################################################
//############################################################################

// Forward declaration.
class CollisionMap;
class Effect;
class EntityArchtype;
class Turret;
class WeaponSlot;

//############################################################################
// Class #####################################################################
//############################################################################

/** Entity represents one physical object that has properties that affect it's
 * mobility, and one game object that has model (or models), collision maps
 * and weapons.
 */
class Entity :
  public EntityBase
{
  protected:
    /** The entity archtype this refers to. */
    EntityArchtype *base_class;

    /** Mass of this (kg). */
    float mass;

    /** Maximum allowed velocity for this object. */
    float max_velocity;

    /** Thurst strength (newton). */
    float thurst[THURSTER_COUNT];

    /** Vertical drag coefficient. */
    float drag_vertical;

    /** Horizontal drag coefficient. */
    float drag_horizontal;

    /** Rotation speed (over z axis) of this. */
    int rotspeed;

    /** Main position. */
    libfhi::Vector2 position;

    /** Movement vector. */
    libfhi::Vector2 movement;

    /** Acceleration vector. */
    libfhi::Vector2 accel;

    /** Angle of rotation (over the z axis). */
    uint16_t angle;

    /** Tilt (bank) of the entity. */
    int tilt;

    /** List of distinct models, not used in reference entities. */
    std::vector<libfhi::PostModel*> models;

    /** List of distinct collision maps. */
    std::vector<CollisionMap*> maps;

    /** List of distinct weapon slots. */
    std::vector<WeaponSlot*> weaponslots;

    /** List of turrets. */
    std::list<Turret*> turrets;

    /** True if object is firing. */
    bool absorb_fire;

    /** Absorb state. */
    int absorb_state;

    /** Wait time until absorb state is usable again. */
    int absorb_wait;

    /** Weapon counters for absorbtion. */
    int absorb_ammo[WEAPON_COUNT];

    /** Current charge. */
    int charge_now;

    /** Previous charge. */
    int charge_last;

    /** AI path list. */
    std::list<libfhi::Vector2> ai_path;

    /** Last AI path search. */
    int ai_last_path_search;

    /** AI target. */
    Entity *ai_target;

    /** Final data chunk consists of multi-usable data for different types. */
    union
    {
      /** Light bydo specific. */
      struct
      {
	/** Angle adders. */
	uint16_t ang_diff[4][3];

	/** Actual angles of the distinct components. */
	uint16_t ang_act[4][3];
      } lightbydo;
    };

  public:
    Entity(EntityArchtype*, int);
    virtual ~Entity();

    // Virtuals.
  public:
    virtual Entity* get_entity();
    virtual void take_damage(int);

    // Private initialization.
  private:
    void add(CollisionMap*);
    void add(libfhi::Mesh*);
    void add(Turret*);
    void add(WeaponSlot*);

  public:
    void absorb_bullet(WeaponEnum);
    void ai();
    void clear_from_terrain();
    void collide_with(CollisionMap::CollisionLine*);
    void collide_with(Entity*);
    void fire();
    void fire_thurst(ThursterEnum, float, float, const libfhi::Vector2&,
	libfhi::Vector2&);
    void fire_turrets();
    void desire(const libfhi::Vector2&, const libfhi::Vector2&);
    void desire(const libfhi::Vector2&, int);
    void draw(const libfhi::Vector2&);
    void input_commands(int*);
    void insert();
    void invoke_death_sparks();
    void normalize_thurst(libfhi::Vector2&);
    bool tick();
    void tilt_return();
    bool try_insert();

    // AI methods.
  protected:
    bool ai_follow_path();
    const libfhi::Vector2& ai_path_lookahead(int);
    libfhi::Vector2 ai_get_allowed_vincinity_point(const libfhi::Vector2&,
	float);
    const libfhi::Vector2& ai_get_path_front() const;
    void ai_search_path(libfhi::Vector2);
    libfhi::Vector2 ai_select_target(bool);
    bool ai_try_aim();
    void ai_battleship();
    void ai_dumbwalker();
    void ai_rtype();

    // Static AI methods.
  public:
    static float ai_aim_lookahead(const libfhi::Vector2&,
	const libfhi::Vector2&, const libfhi::Vector2&, float);
    static int ai_fire_decision(bool, int, std::vector<WeaponSlot*>*);
    static std::vector< std::pair<float, Entity*> >*
      ai_get_proximity_targets(int, size_t, const libfhi::Vector2&,
  	  float);
    static int ai_try_rotate(uint16_t, int, int);

    // Static methods.
  public:
    static float calculate_max_velocity(float, float, float);

    // Inline methods.
  public:
    inline int get_absorb_ammo(WeaponEnum) const;
    inline int get_absorb_state() const;
    inline uint16_t get_angle() const;
    inline EntityArchtype* get_base_class();
    inline int get_charge() const;
    inline const libfhi::Vector2& get_mov() const;
    inline const libfhi::Vector2& get_pos() const;
		inline const float get_mass() const;
    inline std::vector<WeaponSlot*>* get_weaponslots();
    inline void set_absorb_state(int);
    inline void set_angle(uint16_t);
    inline void set_life(int);
    inline void set_mov(const libfhi::Vector2&);
    inline void set_pos(const libfhi::Vector2&);
    inline void set_pos(float, float);

    // Static inline methods.
  public:
    static inline libfhi::Vector2 calculate_accel(libfhi::Vector2&,
      	const libfhi::Vector2&, float, float, float, float, float);
    static inline void apply_physics(libfhi::Vector2&, libfhi::Vector2&,
	const libfhi::Vector2&, float, float, float, float, float, float);
    static inline void evaluate(libfhi::Vector2&, const libfhi::Vector2&,
	libfhi::Vector2&, libfhi::Vector2&, float, float, float, float, float,
	float);
    static inline void apply_simple_physics(libfhi::Vector2&,
	libfhi::Vector2&, libfhi::Vector2&, const libfhi::Vector2&, float,
	float, float, float, float, float);
    static inline float coord_g2r(float);
    static inline float coord_r2g(float);
};

//############################################################################
// Inline methods ############################################################
//############################################################################

/** Get absorbed ammunition of given type.
 * @param op Weapon type.
 */
int Entity::get_absorb_ammo(WeaponEnum op) const
{
  return this->absorb_ammo[op];
}

/** Return the current absorb state of this.
 */
int Entity::get_absorb_state() const
{
  return this->absorb_state;
}

/** Get x rotation angle.
 * @return Angle as uint16_t.
 */
uint16_t Entity::get_angle() const
{
  return this->angle;
}

/** Get the base class archtype of this.
 * @return pointer to archtype.
 */
EntityArchtype* Entity::get_base_class()
{
  return this->base_class;
}

/** Get the charge of given object, this frame.
 */
int Entity::get_charge() const
{
  return this->charge_now;
}

/** Get the movement of this model.
 * @return Movement as a const vector reference.
 */
const libfhi::Vector2& Entity::get_mov() const
{
  return this->movement;
}

/** Get the position of this model.
 * @return Position as a const vector reference.
 */
const libfhi::Vector2& Entity::get_pos() const
{
  return this->position;
}

/** Get the mass of this model.
 * @return Mass as a const float reference.
 */
const float Entity::get_mass() const
{
  return this->mass;
}

/** Get the list of weapon slots in this model.
 * @return Pointer to std::vector.
 */
inline std::vector<WeaponSlot*>* Entity::get_weaponslots()
{
  return &(this->weaponslots);
}

/** Set the absorb state of this entity.
 * @param op New state.
 */
void Entity::set_absorb_state(int op)
{
  this->absorb_state = op;
}

/** Set the angle of this model.
 * @param z Z rotation.
 */
void Entity::set_angle(uint16_t z)
{
  this->angle = z;
}

/** Set the life of this entity.
 * @param op New life value.
 */
void Entity::set_life(int op)
{
  this->life = op;
}

/** Set the movement vector of this.
 * @param op Vector to add.
 */
void Entity::set_mov(const libfhi::Vector2& op)
{
  this->movement = op;
}

/** Replace the position vector of this model.
 * @param op New position.
 */
void Entity::set_pos(const libfhi::Vector2& op)
{
  this->position = op;
}

/** Set new position for this model.
 * @param x X coordinate.
 * @param y Y coordinate.
 */
void Entity::set_pos(float x, float y)
{
  this->position.set(x, y);
}

//############################################################################
// Static methods ############################################################
//############################################################################

libfhi::Vector2 Entity::calculate_accel(libfhi::Vector2 &mov,
    const libfhi::Vector2 &thurst, float cr, float sr, float drag_v,
    float drag_h, float mass)
{
  Terrain *terrain = Terrain::instance;
  
  // take unit vector of direction
  libfhi::Vector2 direction(cr, sr);

  // Calculate unit vector of movement.
  libfhi::Vector2 mov_unit;
  float velocity = mov.length();
  if(velocity <= 0.0f)
  {
    mov_unit.set(0.0f, 0.0f);
  }
  else
  {
    mov_unit = mov / velocity;
  }

  // Calculate drag dot product.
  float draghelp = mov_unit.product_dot(direction);

  // Obtain true drag constant.
  float drag = draghelp * drag_v + (1.0f - draghelp) * drag_h;
	
  // Calculate total drag force in direction of movement
  float dragforce = -0.5f * velocity * velocity * drag * terrain->get_rho();
	
  return (thurst + mov_unit * dragforce) / mass;
}

void Entity::evaluate(libfhi::Vector2 &mov, const libfhi::Vector2 &thurst,
    libfhi::Vector2 &dx, libfhi::Vector2 &dv, float cr, float sr,
    float drag_v, float drag_h, float mass, float timestep)
{
  dx = mov + dv * timestep;
  dv = calculate_accel(dx, thurst, cr, sr, drag_v, drag_h, mass);
}

/** (This method by Warma) Uses runge-kutta integrator for a 4rd degree
 * approximation of the physics engine. General function. Only works in 2
 * dimensions because the game area lacks the third.
 * @param pos Position, modify in-place.
 * @param mov Movement, modify in-place.
 * @param accel Acceleration, modify in-place.
 * @param thurst Read-only thurst.
 * @param cr Cosine of current orientation (rotational direction).
 * @param sr Sine of current orientation (rotational direction).
 * @param drag_v Vertical drag.
 * @param drag_h Horizontal drag.
 * @param mass Mass of object.
 */
void Entity::apply_physics(libfhi::Vector2 &pos, libfhi::Vector2 &mov,
    const libfhi::Vector2 &thurst, float cr, float sr, float drag_v,
    float drag_h, float mass, float maxv)
{
  Terrain *terrain = Terrain::instance;

  //It is neccessary to clamp movement to avoid drag deacceleration divergence
  float velocity = mov.length();
  if(velocity >= maxv)
  {
    mov = (mov/velocity)*maxv;
  }

  // Step 1, store current values into k1
  libfhi::Vector2 dx1 = mov;
  libfhi::Vector2 dv1 = calculate_accel(mov, thurst, cr, sr, drag_v, drag_h,
      mass);

  //std::cout << "Step1v: " << dx1 << " Step1a: " << dv1 << "\n";

  // Step 2, calculate k2 from k1
  libfhi::Vector2 dx2 = dx1;
  libfhi::Vector2 dv2 = dv1;
  evaluate(mov, thurst, dx2, dv2, cr, sr, drag_v, drag_h, mass,
      PHYSICS_STEP_TIME_HALF);

  //std::cout << "Step2v: " << dx2 << " Step2a: " << dv2 << "\n";

  // Step 3, calculate k3 from k2
  libfhi::Vector2 dx3 = dx2;
  libfhi::Vector2 dv3 = dv2;
  evaluate(mov, thurst, dx3, dv3, cr, sr, drag_v, drag_h, mass,
      PHYSICS_STEP_TIME_HALF);

  //std::cout << "Step3v: " << dx3 << " Step3a: " << dv3 << "\n";

  // Step 4, calculate k4 from k3
  libfhi::Vector2 dx4 = dx3;
  libfhi::Vector2 dv4 = dv3;
  evaluate(mov, thurst, dx4, dv4, cr, sr, drag_v, drag_h, mass,
      PHYSICS_STEP_TIME);

  //std::cout << "Step4v: " << dx4 << " Step4a: " << dv4 << "\n";

  // Step 5, add all values and mesure
  libfhi::Vector2 dxdt = (dx1 + dx2 * 2.0f + dx3 * 2.0f + dx4) / 6.0;
  libfhi::Vector2 dvdt = (dv1 + dv2 * 2.0f + dv3 * 2.0f + dv4) / 6.0;
  pos.xf = libfhi::congr(pos.xf + dxdt.xf * PHYSICS_STEP_TIME,
      terrain->get_size_x()); 
  pos.yf = libfhi::congr(pos.yf + dxdt.yf * PHYSICS_STEP_TIME,
      terrain->get_size_y());

  mov += dvdt * PHYSICS_STEP_TIME;
}

/** (This method by Warma) Uses velocity-verlet integrator for 3rd degree
 * approximation of the physics engine. General function. Only works in 2
 * dimensions because the game area lacks the third.
 * @param pos Position, modify in-place.
 * @param mov Movement, modify in-place.
 * @param accel Acceleration, modify in-place.
 * @param thurst Read-only thurst.
 * @param cr Cosine of current orientation (rotational direction).
 * @param sr Sine of current orientation (rotational direction).
 * @param drag_v Vertical drag.
 * @param drag_h Horizontal drag.
 * @param mass Mass of object.
 * @param maxv Maximum velocity.
 */
void Entity::apply_simple_physics(libfhi::Vector2 &pos, libfhi::Vector2 &mov,
    libfhi::Vector2 &accel, const libfhi::Vector2 &thurst, float cr, float sr,
    float drag_v, float drag_h, float mass, float maxv)
{
  Terrain *terrain = Terrain::instance;

  //It is neccessary to clamp movement to avoid drag deacceleration divergence
  float velocity = mov.length();
  if(velocity >= maxv)
  {
    mov = (mov/velocity)*maxv;
  }

  // First step, calculate new position from data from previous frame, also
  // bound the position to the world coordinates via the congruential func.
  pos.xf = libfhi::congr(
      pos.xf + mov.xf * PHYSICS_STEP_TIME + 0.5f * accel.xf *
        PHYSICS_STEP_TIME_SQR,
      terrain->get_size_x());
  pos.yf = libfhi::congr(
      pos.yf + mov.yf * PHYSICS_STEP_TIME + 0.5f * accel.yf *
        PHYSICS_STEP_TIME_SQR,
      terrain->get_size_y());

  // mid-step, calculate new acceleration vector and take a dot product of
  // orientation and direction.
  libfhi::Vector2 direction(cr, sr);

  // Calculate unit vector of movement.
  libfhi::Vector2 mov_unit;
  //velocity = mov.length();
  if(velocity <= 0.0f)
  {
    mov_unit.set(0.0f, 0.0f);
  }
  else
  {
    mov_unit = mov / velocity;
  }

  // Calculate drag dot product.
  float draghelp = mov_unit.product_dot(direction);

  // Obtain true drag constant.
  float drag = draghelp * drag_v + (1.0f - draghelp) * drag_h;

  // Calculate total drag force in direction of movement
  float dragforce = -0.5f * velocity * velocity * drag * terrain->get_rho();

  // Second step, calulate half-velocity from data of previous frame.
  mov += accel * PHYSICS_STEP_TIME_HALF;

  // Apply to acceleration vector
  accel = (thurst + mov_unit * dragforce) / mass;

  // Third step, add new acceleration to the half-velocity to complete cycle
  mov += accel * PHYSICS_STEP_TIME_HALF;
}

/** (This method by Warma) Simplified version of the physics function for
 * simple objects like bullets which do not have differing drag surfaces or
 * means of propulsion. Do not modify this method, modify the method above,
 * then backport the changes here.
 * @param pos Position, modify in-place.
 * @param mov Movement, modify in-place.
 * @param accel Acceleration, modify in-place.
 * @param drag Precalculated drag constant.
 * @param mass Mass of object.
 */
/*void Entity::apply_simple_physics(libfhi::Vector2 &pos, libfhi::Vector2 &mov,
    libfhi::Vector2 &accel, float drag, float mass)
{
  Terrain *terrain = Terrain::instance;

  // First step, calculate new position from data from previous frame, also
  // bound the position to the world coordinates via the congruential func.
  pos.xf = libfhi::congr(
      pos.xf + mov.xf * PHYSICS_STEP_TIME + 0.5f * accel.xf *
        PHYSICS_STEP_TIME_SQR,
      terrain->get_size_x());
  pos.yf = libfhi::congr(
      pos.yf + mov.yf * PHYSICS_STEP_TIME + 0.5f * accel.yf *
        PHYSICS_STEP_TIME_SQR,
      terrain->get_size_y());

  // Second step, calulate half-velocity from the same data.
  mov += accel * PHYSICS_STEP_TIME_HALF;

  // Calculate unit vector of movement.
  libfhi::Vector2 mov_unit;
  float velocity = mov.length();
  if(velocity <= 0.0f)
  {
    mov_unit.set(0.0f, 0.0f);
  }
  else
  {
    mov_unit = mov / velocity;
  }

  // calcultate total drag force in direction of movement.
  float dragforce = -0.5f * velocity * velocity * drag * terrain->get_rho();

  // Apply to acceleration vector.
  accel = mov_unit * dragforce / mass;

  // Fourth step, add new acceleration to the half-velocity to complete cycle.
  mov += accel * PHYSICS_STEP_TIME_HALF;
}*/

/** Transform a game coordinate into a real-world (meter) coordinate.
 * @param op Game coordinate.
 * @return Real-world coordinate.
 */
float Entity::coord_g2r(float op)
{
  return op * PHYSICS_STEP_SPACE;
}

/** Transform a real world coordinate (meter) into a game coordinate.
 * @param op Real-world coordinate.
 * @return Game coordinate.
 */
float Entity::coord_r2g(float op)
{
  return op * PHYSICS_STEP_SPACE_INV;
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

