#ifndef libfhi_font_include
#define libfhi_font_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_canonicaldb.h"

#include HASHMAP_INCLUDE

#ifdef LIBFHI_OPENGL
#include LIBFHI_GLEW_H
#endif

namespace libfhi {

//############################################################################
// Class #####################################################################
//############################################################################

/** Glyph represents one drawable character.
 */
class Glyph
{
#ifdef LIBFHI_OPENGL
  private: // Only include texture name if OpenGL compilation is on.
    GLuint texture_name;
    float tmx, tmy;
#endif

  private:
    int w, h, ax, ay, lt, tp;
    uint8_t *data;

  public:
    Glyph(void*);
    ~Glyph();

    // Conditional methods.
#ifdef LIBFHI_OPENGL
  public:
    GLuint generate_texture();
#endif

    // Inline methods.
  public:
    inline int get_advance_x() const;
    inline int get_advance_y() const;
    inline uint8_t* get_data();
    inline int get_h() const;
    inline int get_left() const;
    inline int get_top() const;
    inline int get_w() const;

    // Conditional inline methods.
#ifdef LIBFHI_OPENGL
    inline GLuint get_texture() const;
    inline float get_tmx() const;
    inline float get_tmy() const;
#endif
};

/** This class is made to contain one font (FT_Face) and cache all glyphs in
 * it. This class is not made for reuse, when done with one font, delete it
 * and create a new one.
 */
class Font :
  public CanonicalDB<Font>
{
  private:
    /** The font face for FreeType (actually FT_FACE, remember typecast). */
    void *face;

    /** Map to store glyphs in. */
    HASHMAP<int, Glyph*> glyphs;

  public:
    inline Font();
    inline ~Font();

    // Methods.
  public:
    void clear_cache();
    Glyph* get_glyph(int);
    int get_string_width(const char*);
    bool load(const char*);
    bool set_size(int, int = 0);
};

//############################################################################
// Functions #################################################################
//############################################################################

// Font handling functions.
bool font_init();
Font* font_new(const char*);
bool font_quit();

//############################################################################
// Inline functions of Glyph #################################################
//############################################################################

/** Get the color data.
 * @return Pointer to unsigned byte array.
 */
uint8_t* Glyph::get_data()
{
  return this->data;
}

/** Get the width.
 * @return The width of this in bytes.
 */
int Glyph::get_w() const
{
  return this->w;
}

/** Get the heigth.
 * @return The height of this in bytes.
 */
int Glyph::get_h() const
{
  return this->h;
}

/** Get the left border of glyph.
 * @return The left border.
 */
int Glyph::get_left() const
{
  return this->lt;
}

/** Get the top border of glyph.
 * @return The top border.
 */
int Glyph::get_top() const
{
  return this->tp;
}

/** Get the advancment in pixels (x direction)
 * @return Pixel advancment in x direction.
 */
int Glyph::get_advance_x() const
{
  return this->ax;
}

/** Get the advancment in pixels (y direction)
 * @return Pixel advancment in y direction.
 */
int Glyph::get_advance_y() const
{
  return this->ay;
}

//############################################################################
// Conditional inline methods of Glyph #######################################
//############################################################################

#ifdef LIBFHI_OPENGL

/** Get the texture name of this Glyph.
 * @return Texture name or 0 if no texture reserved.
 */
GLuint Glyph::get_texture() const
{
  return this->texture_name;
}

/** Get the maximum texture coordinate in x.
 * @return Maximum coord, between [0, 1].
 */
float Glyph::get_tmx() const
{
  return this->tmx;
}

/** Get the maximum texture coordinate in y.
 * @return Maximum coord, between [0, 1].
 */
float Glyph::get_tmy() const
{
  return this->tmy;
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif

