#ifndef libfhi_matvec_include
#define libfhi_matvec_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_fixed.h"

namespace libfhi {

//############################################################################
// Definet ###################################################################
//############################################################################

// Forward declaration
class Matrix44;
class Vector3;

//############################################################################
// Luokka ####################################################################
//############################################################################

/** Vector2 is a simplified class for 2d vector math. For detailed
 * descriptions, see Vector3.
 */
class Vector2
{
  public:
    union
    {
      // Array representation.
      float f[2];
      
      struct
      {
	// Coordinates
	union
	{
	  float xf;
	  Fixed xi;
	};
	union
	{
	  float yf;
	  Fixed yi;
	};
      };
    };

  public:
#ifdef LIBFHI_DEBUG
    DEBUG_PRINT_PROTOTYPE(Vector2)
#endif

  public:
    /* Constructors. */
    inline Vector2();
    inline Vector2(const Vector2&);
    inline Vector2(float, float);
    inline Vector2(int, int);
    inline ~Vector2();
    
    // Other methods
    inline float* get_fv();
    inline const float* get_fv() const;
    inline float get_planar_angle(const Vector2&) const;
    inline bool is_null() const;
    inline float length() const;
    inline float length_sqr() const;
    inline void null();
    inline float product_dot(const Vector2&) const;
    inline void rotate(const libfhi::Vector2&, float, float);
    inline void rotate_translate(const libfhi::Vector2&, float, float,
	const libfhi::Vector2&);
    inline void round();
    inline void set(float, float);
    inline void set(int, int);
    inline void unit();

    // Operators
    inline const Vector2 operator+(const Vector2&) const;
    inline const Vector2 operator-(const Vector2&) const;
    inline const Vector2 operator-() const;
    inline const Vector2 operator*(float) const;
    inline const Vector2 operator/(float) const;
    inline void operator+=(const Vector2&);
    inline void operator-=(const Vector2&);
    inline void operator*=(float);
    inline void operator/=(float);
    inline bool operator==(const Vector2&) const;
    inline bool operator!=(const Vector2&) const;
    inline void operator=(const Vector2&);
};

/** Vector and matrix classes here are aligned as follows:
 *
 * Matrix:            Vector:
 * 00  04  08  12     0 (x)
 * 01  05  09  13     1 (y)
 * 02  06  10  14     2 (z)
 * 04  07  11  15
 *
 * In actuality, this is the representation of 4x4 matrices enabling all
 * euclidian three-dimensional transformations in OpenGL column-major form,
 * where saving or loading the matrices into the OpenGL matrix stack would
 * require no alteration of the order of the data.
 *
 * If the fourth component of vectors would be used, it would be treated as
 * 1, the same manner as matrix component 15 will mostly be 1 and components
 * 4, 7 and 11 are likely to be 0. However, in the case of matrices, these
 * optimizations are not assumed for the sake of generalization.
 *
 * As another optimization note, vectors and matrices are in no point assumed
 * to be real structures, but are treated as primitive types that the user
 * may perform mathematic operations with at will. This might be ineffective,
 * but it is intended to be written as inline operations for stack variables,
 * that should (as far as my knowledge of compiler technology is concerned)
 * enable the compiler to perform a lot of optimizations in register coloring
 * and coalescing of moves.
 *
 * And, oh yeah, it makes writing the utilizing code a lot more easier than
 * it theoretically could make the output slower.
 *
 * As with Color, if an operation can affect a varying number of components,
 * the user must append the method call with this number (3 or 2) to prevent
 * accidental erraneous resource use.
 */
class Vector3
{
  public:
    union
    {
      // Array representation.
      float f[3];
      
      struct
      {
	// Coordinates
	union
	{
	  float xf;
	  Fixed xi;
	};
	union
	{
	  float yf;
	  Fixed yi;
	};
	union
	{
	  float zf;
	  Fixed zi;
	};
      };
    };

  public:
#ifdef LIBFHI_DEBUG
    DEBUG_PRINT_PROTOTYPE(Vector3)
#endif

  public:
    /* Constructors. */
    inline Vector3();
    inline Vector3(const Vector3&);
    inline Vector3(float, float, float);
    inline Vector3(int, int, int);
    inline ~Vector3();
    
    // Other methods
    inline void average(const Vector3&, const Vector3&);
    inline void average(const Vector3&, const Vector3&, const Vector3&);
    inline void average(const Vector3&, const Vector3&, const Vector3&,
	const Vector3&);
    inline float clip_value_x(const Vector3&, const Vector3&, float);
    inline float clip_value_y(const Vector3&, const Vector3&, float);
    inline float clip_value_z(const Vector3&, const Vector3&, float);
    inline float clip3_x(const Vector3&, const Vector3&, float);
    inline float clip3_y(const Vector3&, const Vector3&, float);
    inline float clip3_z(const Vector3&, const Vector3&, float);
    inline float clip2_x(const Vector3&, const Vector3&, float);
    inline float clip2_y(const Vector3&, const Vector3&, float);
    inline float* get_fv();
    inline const float* get_fv() const;
    inline bool is_null() const;
    inline bool is_same_direction(const Vector3&) const;
    inline float length() const;
    inline void null();
    inline void product_cross(const Vector3&, const Vector3&);
    inline float product_dot(const Vector3&) const;
    inline void round3();
    inline void round2();
    inline void set(float, float, float);
    inline void set(int, int, int);
    inline void set_roundable(int, int);
    inline void set_roundable(int, int, int);
    inline void unit();

    // Operators
    inline const Vector3 operator+(const Vector3&) const;
    inline const Vector3 operator-(const Vector3&) const;
    inline const Vector3 operator*(float) const;
    inline const Vector3 operator/(float) const;
    inline void operator+=(const Vector3&);
    inline void operator-=(const Vector3&);
    inline void operator*=(float);
    inline void operator/=(float);
    inline bool operator==(const Vector3&) const;
    inline bool operator!=(const Vector3&) const;
    inline void operator=(const Vector3&);
};

/** Matrix class, 4x4 OpenGL clone. For description, see Vector.
 */
class Matrix44
{
  public:
    float f[16];

  public:
#ifdef LIBFHI_DEBUG
    DEBUG_PRINT_PROTOTYPE(Matrix44)
#endif

  public:
    /* Constructors. */
    inline Matrix44();
    inline Matrix44(float, float, float, float, float, float, float, float,
	float, float, float, float, float, float, float, float);
    inline Matrix44(const Matrix44& op);
    inline ~Matrix44();

    // Other methods
    inline const Vector3 get_fw() const;
    inline const float* get_glfv() const;
    inline const Vector3 get_pos() const;
    inline const Vector3 get_rt() const;
    inline const Vector3 get_up() const;
    inline void identity();
    inline void null();
    inline void set(float, float, float, float, float, float, float, float,
	float, float, float, float, float, float, float, float);

    // Non-inline
    void lookat(const Vector3&, const Vector3&, const Vector3&);
    void normalize();
    void rotation_rel(const Matrix44&, uint16_t, uint16_t, uint16_t);
    void rotation_zxy33(uint16_t, uint16_t, uint16_t);
    void rotation_zxy(uint16_t, uint16_t, uint16_t);
    void rotation_zxy(const uint16_t*);
    void rotation_zxy_translation(const uint16_t*, const Vector3&);
    void rotation_zyx33(uint16_t, uint16_t, uint16_t);
    void rotation_zyx_translation(const uint16_t*, const Vector3&);
    void translation_13(float, float, float);
    void translation(float, float, float);
    void translation(const Vector3&);
    void transpose(const Matrix44&);
    void view_matrix(const Matrix44&);

    // Operators
    inline const Matrix44 operator+(const Matrix44&) const;
    inline const Matrix44 operator-(const Matrix44&) const;
    inline const Matrix44 operator*(float) const;
    inline const Matrix44 operator*(const Matrix44&) const;
    inline const Vector3 operator*(const Vector3&) const;
    inline void operator+=(const Matrix44&);
    inline void operator-=(const Matrix44&);
    inline void operator*=(float);
    inline void operator=(const Matrix44&);
};

//############################################################################
// Vector2 constructor #######################################################
//############################################################################

/** Default constructor.
 */
Vector2::Vector2()
{
  // Do nothing.
}

/** Default destructor.
 */
Vector2::~Vector2()
{
  // Do nothing.
}

/** Copy constructor.
 * @param op Vector2 to copy.
 */
Vector2::Vector2(const Vector2& op)
{
  (*this) = op;
}

/** Initializing constructor (float).
 * @param x X component.
 * @param y Y component.
 */
Vector2::Vector2(float x, float y)
{
  this->set(x, y);
}

/** Initializing constructor (int).
 * @param x X component.
 * @param y Y component.
 */
Vector2::Vector2(int x, int y)
{
  this->set(x, y);
}

//############################################################################
// Vector3 set / get #########################################################
//############################################################################

/** Get floating point array of this.
 * @return contents as a float array.
 */
float* Vector2::get_fv()
{
  return this->f;
}

/** Get floating point array of this (const version).
 * @return contents as a float array.
 */
const float* Vector2::get_fv() const
{
  return this->f;
}

/** Get the planar angle of this and destination vector, this is used as the
 * base, destination vector is the vector to calculate difference against.
 * Both vectors must be unit vectors.
 * @param op Vector to calculate the angle to.
 * @return Angle between two vectors in bounds [0, 2 * PI].
 */
float Vector2::get_planar_angle(const Vector2 &op) const
{
  libfhi::Vector2 temp;

  temp.rotate(*this, -op.xf, -op.yf);

  return atan2f(temp.yf, temp.xf);
}

/** Empty this vector.
 */
void Vector2::null()
{
  this->xf = this->yf = 0.0f;
}

/** Round this vector to fixed point.
 */
void Vector2::round()
{
  this->xi.set_roundable(this->xf);
  this->yi.set_roundable(this->yf);
}

/** Set the values of this (float).
 * @param x X component.
 * @param y Y component.
 */
void Vector2::set(float x, float y)
{
  this->xf = x;
  this->yf = y;
}

/** Set the values of this (int).
 * @param x X component.
 * @param y Y component.
 */
void Vector2::set(int x, int y)
{
  this->xi = Fixed::from(x);
  this->yi = Fixed::from(y);
}

/** Assignment operator.
 * @param rhs Right-hand side operand.
 */
void Vector2::operator=(const Vector2 &rhs)
{
  this->xi = rhs.xi;
  this->yi = rhs.yi;
}

//############################################################################
// Vector2 probe #############################################################
//############################################################################

/** Tell if this vector is empty. Not for realtime really.
 * @return True if yes, false if no.
 */
bool Vector2::is_null() const
{
  return (fabsf(this->xf) + fabsf(this->yf) == 0.0f);
}

/** Test the equality of vectors.
 * @return True if equal, false if inequal.
 */
bool Vector2::operator==(const Vector2 &rhs) const
{
  if((this->xi == rhs.xi) && (this->yi == rhs.yi))
  {
    return true;
  }
  return false;
}

/** Test the inequality of vectors.
 * @return True if inequal, false if equal.
 */
bool Vector2::operator!=(const Vector2 &rhs) const
{
  if((this->xi != rhs.xi) || (this->yi != rhs.yi))
  {
    return true;
  }
  return false;
}

//############################################################################
// Vector2 math ##############################################################
//############################################################################

/** Calculate the length of this.
 * @return Length as float.
 */
float Vector2::length() const
{
  return sqrtf(this->length_sqr());
}

/** Calculate the length of this in squared form (for speed) when it needs not
 * be exact.
 * @return Squared length as float.
 */
float Vector2::length_sqr() const
{
  return this->xf * this->xf + this->yf * this->yf;
}

/** Calculate the dot product of this vector with another vector.
 * @return Result as float.
 */
float Vector2::product_dot(const Vector2 &op) const
{
  return this->xf * op.xf + this->yf * op.yf;
}

/** Rotate target vector with given cosine and sine, save to this.
 * @param src Source vector to rotate.
 * @param cr Rotation cosine.
 * @param sr Rotation sine.
 */
void Vector2::rotate(const libfhi::Vector2 &src, float cr, float sr)
{
  this->xf = src.xf * cr - src.yf * sr;
  this->yf = src.xf * sr + src.yf * cr;
}

/** Rotate target vector with given cosine and sine, then add another target
 * vector, save to this.
 * @param src Source vector to rotate.
 * @param cr Rotation cosine.
 * @param sr Rotation sine.
 * @param trans Translation vector.
 */
void Vector2::rotate_translate(const libfhi::Vector2 &src, float cr, float sr,
    const libfhi::Vector2 &trans)
{
  this->xf = src.xf * cr - src.yf * sr + trans.xf;
  this->yf = src.xf * sr + src.yf * cr + trans.yf;
}

/** Unit this vector. Will crash if length = 0.
 */
void Vector2::unit()
{
  float len = this->length();
  this->xf /= len;
  this->yf /= len;
}

/** Get sum of this and another vector.
 * @param rhs Right-hand side operand.
 * @return Sum vector.
 */
const Vector2 Vector2::operator+(const Vector2 &rhs) const
{
  return Vector2(this->xf + rhs.xf,
      this->yf + rhs.yf);
}

/** Get difference of this and another vector.
 * @param rhs Right-hand side operand.
 * @return Difference vector.
 */
const Vector2 Vector2::operator-(const Vector2 &rhs) const
{
  return Vector2(this->xf - rhs.xf,
      this->yf - rhs.yf);
}

/** Unary minus of Vector2.
 * @return Inverse vector.
 */
const Vector2 Vector2::operator-() const
{
  return Vector2(-this->xf, -this->yf);
}

/** Get multiplification of this by a variable.
 * @param rhs Multiplying constant.
 * @return Scaled vector.
 */
const Vector2 Vector2::operator*(float rhs) const
{
  return Vector2(this->xf * rhs, this->yf * rhs);
}

/** Divide this by a variable.
 * @param rhs Dividing operand
 * @return Scaled vector.
 */
const Vector2 Vector2::operator/(float rhs) const
{
  return Vector2(this->xf / rhs, this->yf / rhs);
}

/** Increment this vector by given vector.
 * @param rhs Right-hand side operand.
 */
void Vector2::operator+=(const Vector2 &rhs)
{
  this->xf += rhs.xf;
  this->yf += rhs.yf;
}

/** Substract this vector by given vector.
 * @param rhs Right-hand side operand.
 */
void Vector2::operator-=(const Vector2 &rhs)
{
  this->xf -= rhs.xf;
  this->yf -= rhs.yf;
}

/** Multiply this vector by given variable.
 * @param rhs Right-hand side operand.
 */
void Vector2::operator*=(float rhs)
{
  this->xf *= rhs;
  this->yf *= rhs;
}

/** Divide this vector by given variable.
 * @param rhs Right-hand side operand.
 */
void Vector2::operator/=(float rhs)
{
  this->xf /= rhs;
  this->yf /= rhs;
}

//############################################################################
// Vector3 constructor #######################################################
//############################################################################

/** Default constructor.
 */
Vector3::Vector3()
{
  // Do nothing.
}

/** Default destructor.
 */
Vector3::~Vector3()
{
  // Do nothing.
}

/** Copy constructor.
 * @param op Vector to copy.
 */
Vector3::Vector3(const Vector3& op)
{
  (*this) = op;
}

/** Initialization constructor (float).
 * @param x X component.
 * @param y Y component.
 * @param z Z component.
 */
Vector3::Vector3(float x, float y, float z)
{
  this->set(x, y, z);
}

/** Initialization constructor (int).
 * @param x X component.
 * @param y Y component.
 * @param z Z component.
 */
Vector3::Vector3(int x, int y, int z)
{
  this->set(x, y, z);
}

//############################################################################
// Vector3 average ###########################################################
//############################################################################

/** Calculate average of targets, store to this.
 * @param op1 First vector.
 * @param op2 Second vector.
 */
void Vector3::average(const Vector3& op1, const Vector3& op2)
{
  this->xf = (op1.xf + op2.xf) / 2.0f;
  this->yf = (op1.yf + op2.yf) / 2.0f;
  this->zf = (op1.zf + op2.zf) / 2.0f;
}

/** Calculate average of targets, store to this.
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param op3 Third vector.
 */
void Vector3::average(const Vector3& op1, const Vector3& op2,
    const Vector3& op3)
{
  this->xf = (op1.xf + op2.xf + op3.xf) / 3.0f;
  this->yf = (op1.yf + op2.yf + op3.yf) / 3.0f;
  this->zf = (op1.zf + op2.zf + op3.zf) / 3.0f;
}

/** Calculate average of targets, store to this.
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param op3 Third vector.
 * @param op4 Fourth vector.
 */
void Vector3::average(const Vector3& op1, const Vector3& op2,
    const Vector3& op3, const Vector3& op4)
{
  this->xf = (op1.xf + op2.xf + op3.xf + op4.xf) / 4.0f;
  this->yf = (op1.yf + op2.yf + op3.yf + op4.yf) / 4.0f;
  this->zf = (op1.zf + op2.zf + op3.zf + op4.zf) / 4.0f;
}

//############################################################################
// Vector3 clip ##############################################################
//############################################################################

/** Calculator for coming clippings. Calculates a value between 0.0f and 1.0f
 * that is the relative distance from first vector to the second and equals
 * the clip value (between them) in the given axis.
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param value The value to clip to.
 * @return The relative clip value.
 */
float Vector3::clip_value_x(const Vector3& op1, const Vector3& op2, float value)
{
  return (value - op1.xf) / (op2.xf - op1.xf);
}

/** Clip value for y (see clip_value_x for description).
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param value The value to clip to.
 * @return The relative clip value.
 */
float Vector3::clip_value_y(const Vector3& op1, const Vector3& op2, float value)
{
  return (value - op1.yf) / (op2.yf - op1.yf);
}

/** Clip value for z (see clip_value_x for description).
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param value The value to clip to.
 * @return The relative clip value.
 */
float Vector3::clip_value_z(const Vector3& op1, const Vector3& op2, float value)
{
  return (value - op1.zf) / (op2.zf - op1.zf);
}

/** 3d clipper in x. Calling vector is set to the point where the line formed
 * by two given vectors would intersect the x plane at x = value. This point
 * may or may not be between the vectors.
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param value Clip value in the plane.
 * @return The relative clip value.
 */
float Vector3::clip3_x(const Vector3& op1, const Vector3& op2, float value)
{
  float percent = clip_value_x(op1, op2, value),
	dy = op2.yf - op1.yf,
	dz = op2.zf - op1.zf;

  this->xf = value;
  this->yf = dy * percent + op1.yf; 
  this->zf = dz * percent + op1.zf; 
  return percent;
}

/** 3d clipper in y (for description, see clip_x).
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param value Clip value in Y.
 * @return The relative clip value.
 */
float Vector3::clip3_y(const Vector3& op1, const Vector3& op2, float value)
{
  float percent = clip_value_y(op1, op2, value),
	dx = op2.xf - op1.xf,
	dz = op2.zf - op1.zf;

  this->xf = dx * percent + op1.xf; 
  this->yf = value;
  this->zf = dz * percent + op1.zf; 
  return percent;
}

/** 3d clipper in z (for description, see clip_x).
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param value Clip value in Z.
 * @return The relative clip value.
 */
float Vector3::clip3_z(const Vector3& op1, const Vector3& op2, float value)
{
  float percent = clip_value_z(op1, op2, value),
	dx = op2.xf - op1.xf,
	dy = op2.yf - op1.yf;

  this->xf = dx * percent + op1.xf; 
  this->yf = dy * percent + op1.yf; 
  this->zf = value;
  return percent;
}

/** 2d clipper in x. Works like the 3d clipper in x, but ignores depth
 * component, used when filling in 2d without depth checking.
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param value Clip value in plane.
 * @return The relative clip value.
 */
float Vector3::clip2_x(const Vector3& op1, const Vector3& op2, float value)
{
  float percent = clip_value_x(op1, op2, value),
	dy = op2.yf - op1.yf;

  this->xf = value;
  this->yf = dy * percent + op1.yf;
  return percent;
}

/** 2d clipper in Y (for description, see clip2_x).
 * @param op1 First vector.
 * @param op2 Second vector.
 * @param value Clip value in plane.
 * @return The relative clip value.
 */
float Vector3::clip2_y(const Vector3& op1, const Vector3& op2, float value)
{
  float percent = clip_value_y(op1, op2, value),
	dx = op2.xf - op1.xf;

  this->xf = dx * percent + op1.xf;
  this->yf = value;
  return percent;
}

//############################################################################
// Vector3  misc #############################################################
//############################################################################

/** Round vector from float to fixed.
 */
void Vector3::round3()
{
  this->xi.set_roundable(xf);
  this->yi.set_roundable(yf);
  this->zi.set_roundable(zf);
}

/** Round the first two components of vector from float to fixed.
 */
void Vector3::round2()
{
  this->xi.set_roundable(xf);
  this->yi.set_roundable(yf);
}

/** Initialize with roundable components.
 * @param x X coordinate.
 * @param y Y coordinate.
 */
void Vector3::set_roundable(int x, int y)
{
  this->xi.set_roundable(x);
  this->yi.set_roundable(y);
}

/** Initialize with roundable components.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param z Z coordinate.
 */
void Vector3::set_roundable(int x, int y, int z)
{
  this->xi.set_roundable(x);
  this->yi.set_roundable(y);
  this->zi.set_roundable(z);
}

//############################################################################
// Vector3 get / set #########################################################
//############################################################################

/** Return the floating point values as a pointer.
 * @return Floating point pointer.
 */
float* Vector3::get_fv()
{
  return this->f;
}

/** Return the floating point values as a pointer (const version).
 * @return Floating point pointer.
 */
const float* Vector3::get_fv() const
{
  return this->f;
}

/** Empty this vector.
 */
void Vector3::null()
{
  this->xi.set_null();
  this->yi.set_null();
  this->zi.set_null();
}

/** Assign floating point values to this vector.
 * @param x [in] New x component.
 * @param y [in] New y component.
 * @param z [in] New z component.
 */
void Vector3::set(float x, float y, float z)
{
  this->xf = x;
  this->yf = y;
  this->zf = z;
}
  
/** Assign integer values to this vector.
 * @param x [in] New x component.
 * @param y [in] New y component.
 * @param z [in] New z component.
 */
void Vector3::set(int x, int y, int z)
{
  this->xi = Fixed::from(x);
  this->yi = Fixed::from(y);
  this->zi = Fixed::from(z);
}

/** Assignment operator.
 * @param rhs [in] Right-hand side of the operation.
 */
void Vector3::operator=(const Vector3& rhs)
{
  this->xi = rhs.xi;
  this->yi = rhs.yi;
  this->zi = rhs.zi;
}

//############################################################################
// Vector3 comparison ########################################################
//############################################################################

/** Test the equality of vectors.
 * @return True if equal, false if inequal.
 */
bool Vector3::operator==(const Vector3& rhs) const
{
  if((this->xi == rhs.xi) && (this->yi == rhs.yi) && (this->zi == rhs.zi))
    return true;
  return false;
}

/** Test the inequality of vectors.
 * @return True if inequal, false if equal.
 */
bool Vector3::operator!=(const Vector3& rhs) const
{
  if((this->xi != rhs.xi) || (this->yi != rhs.yi) || (this->zi != rhs.zi))
    return true;
  return false;
}

/** Tell if this vector is a null vector.
 * Somewhat time consuming method, but has to be done this way, do not use
 * in realtime.
 * @return True if yes, false if no.
 */
bool Vector3::is_null() const
{
  return (fabs(this->xf) + fabs(this->yf) + fabs(this->zf) == 0.0f);
}

/** Tell if this vector is (about) the same direction as given vector.
 * Both vectors must be unit vectors.
 * @param op Vector to compare to.
 * @return True if yes, false if no.
 */
bool Vector3::is_same_direction(const Vector3& op) const
{
  static const float TOLERANCE = 0.99f;

  return (this->product_dot(op) > TOLERANCE);
}

//############################################################################
// Vector3 math ##############################################################
//############################################################################

/** Calculate the length of this.
 * @return Length as float.
 */
float Vector3::length() const
{
  return sqrt(this->xf * this->xf + this->yf * this->yf +
      this->zf * this->zf);
}

/** Assign cross product of two vectors to this.
 * @param op1 First vector.
 * @param op2 Second vector.
 */
void Vector3::product_cross(const Vector3& op1, const Vector3& op2)
{
  this->xf = op1.yf * op2.zf - op1.zf * op2.yf;
  this->yf = op1.zf * op2.xf - op1.xf * op2.zf;
  this->zf = op1.xf * op2.yf - op1.yf * op2.xf;
}

/** Get the cross product of this and given vector.
 * @param op Vector to calculate cross product with.
 */
float Vector3::product_dot(const Vector3& op) const
{
  return op.xf * this->xf + op.yf * this->yf + op.zf * this->zf;
}

/** Set this vector to a length of one. Will crash if length is zero.
 */
void Vector3::unit()
{
  float len = 1.0f / this->length();
  xf *= len;
  yf *= len;
  zf *= len;
}

/** Get sum of this and another vector.
 * @param rhs Right-hand side operand.
 * @return Sum vector.
 */
const Vector3 Vector3::operator+(const Vector3& rhs) const
{
  return Vector3(this->xf + rhs.xf,
      this->yf + rhs.yf,
      this->zf + rhs.zf);
}

/** Get difference of this and another vector.
 * @param rhs Right-hand side operand.
 * @return Difference vector.
 */
const Vector3 Vector3::operator-(const Vector3 &rhs) const
{
  return Vector3(this->xf - rhs.xf,
      this->yf - rhs.yf,
      this->zf - rhs.zf);
}

/** Get multiplification of this by a variable.
 * @param rhs Multiplying constant.
 * @return Scaled vector.
 */
const Vector3 Vector3::operator*(float rhs) const
{
  return Vector3(this->xf * rhs, this->yf * rhs, this->zf * rhs);
}

/** Divide this by a variable.
 * @param rhs Dividing operand
 * @return Scaled vector.
 */
const Vector3 Vector3::operator/(float rhs) const
{
  return Vector3(this->xf / rhs, this->yf / rhs, this->zf / rhs);
}

/** Increment this vector by given vector.
 * @param rhs Right-hand side operand.
 */
void Vector3::operator+=(const Vector3 &rhs)
{
  this->xf += rhs.xf;
  this->yf += rhs.yf;
  this->zf += rhs.zf;
}

/** Substract this vector by given vector.
 * @param rhs Right-hand side operand.
 */
void Vector3::operator-=(const Vector3 &rhs)
{
  this->xf -= rhs.xf;
  this->yf -= rhs.yf;
  this->zf -= rhs.zf;
}

/** Multiply this vector by given variable.
 * @param rhs Right-hand side operand.
 */
void Vector3::operator*=(float rhs)
{
  this->xf *= rhs;
  this->yf *= rhs;
  this->zf *= rhs;
}

/** Divide this vector by given variable.
 * @param rhs Right-hand side operand.
 */
void Vector3::operator/=(float rhs)
{
  this->xf /= rhs;
  this->yf /= rhs;
  this->zf /= rhs;
}

//############################################################################
// Matrix44 constructor ######################################################
//############################################################################

Matrix44::Matrix44()
{
}

Matrix44::~Matrix44()
{
}

Matrix44::Matrix44(float f00, float f01, float f02, float f03, float f04,
    float f05, float f06, float f07, float f08, float f09, float f10,
    float f11, float f12, float f13, float f14, float f15)
{
  this->set(f00, f01, f02, f03, f04, f05, f06, f07, f08, f09, f10, f11, f12,
      f13, f14, f15);
}

Matrix44::Matrix44(const Matrix44& op)
{
  (*this) = op;
}

//############################################################################
// Matrix44 get / set ########################################################
//############################################################################

const Vector3 Matrix44::get_fw() const
{
  return Vector3(this->f[2], this->f[6], this->f[10]);
}

const float* Matrix44::get_glfv() const
{
  return this->f;
}

const Vector3 Matrix44::get_pos() const
{
  return Vector3(this->f[12], this->f[13], this->f[14]);
}

const Vector3 Matrix44::get_rt() const
{
  return Vector3(this->f[0], this->f[4], this->f[8]);
}

const Vector3 Matrix44::get_up() const
{
  return Vector3(this->f[1], this->f[5], this->f[9]);
}

void Matrix44::identity()
{
  this->f[0] = this->f[5] = this->f[10] = this->f[15] = 1.0f;
  this->f[1] = this->f[2] = this->f[3] = this->f[4] = this->f[6] =
    this->f[7] = this->f[8] = this->f[9] = this->f[11] = this->f[12] =
    this->f[13] = this->f[14] = 0.0f;
}

void Matrix44::null()
{
  this->f[0] = this->f[1] = this->f[2] = this->f[3] = this->f[4] =
    this->f[5] = this->f[6] = this->f[7] = this->f[8] = this->f[9] =
    this->f[10] = this->f[11] = this->f[12] = this->f[13] = this->f[14] =
    this->f[15] = 0.0f;
}

void Matrix44::set(float f00, float f01, float f02, float f03, float f04,
    float f05, float f06, float f07, float f08, float f09, float f10,
    float f11, float f12, float f13, float f14, float f15)
{
  this->f[0] = f00;
  this->f[1] = f01;
  this->f[2] = f02;
  this->f[3] = f03;
  this->f[4] = f04;
  this->f[5] = f05;
  this->f[6] = f06;
  this->f[7] = f07;
  this->f[8] = f08;
  this->f[9] = f09;
  this->f[10] = f10;
  this->f[11] = f11;
  this->f[12] = f12;
  this->f[13] = f13;
  this->f[14] = f14;
  this->f[15] = f15;
}

void Matrix44::operator=(const Matrix44& op)
{
  this->f[0] = op.f[0];
  this->f[1] = op.f[1];
  this->f[2] = op.f[2];
  this->f[3] = op.f[3];
  this->f[4] = op.f[4];
  this->f[5] = op.f[5];
  this->f[6] = op.f[6];
  this->f[7] = op.f[7];
  this->f[8] = op.f[8];
  this->f[9] = op.f[9];
  this->f[10] = op.f[10];
  this->f[11] = op.f[11];
  this->f[12] = op.f[12];
  this->f[13] = op.f[13];
  this->f[14] = op.f[14];
  this->f[15] = op.f[15];
}

//############################################################################
// Matrix44 math #############################################################
//############################################################################

const Matrix44 Matrix44::operator+(const Matrix44& op) const
{
  return Matrix44(this->f[0] + op.f[0],
      this->f[1] + op.f[1],
      this->f[2] + op.f[2],
      this->f[3] + op.f[3],
      this->f[4] + op.f[4],
      this->f[5] + op.f[5],
      this->f[6] + op.f[6],
      this->f[7] + op.f[7],
      this->f[8] + op.f[8],
      this->f[9] + op.f[9],
      this->f[10] + op.f[10],
      this->f[11] + op.f[11],
      this->f[12] + op.f[12],
      this->f[13] + op.f[13],
      this->f[14] + op.f[14],
      this->f[15] + op.f[15]);
}

const Matrix44 Matrix44::operator-(const Matrix44& op) const
{
  return Matrix44(this->f[0] - op.f[0],
      this->f[1] - op.f[1],
      this->f[2] - op.f[2],
      this->f[3] - op.f[3],
      this->f[4] - op.f[4],
      this->f[5] - op.f[5],
      this->f[6] - op.f[6],
      this->f[7] - op.f[7],
      this->f[8] - op.f[8],
      this->f[9] - op.f[9],
      this->f[10] - op.f[10],
      this->f[11] - op.f[11],
      this->f[12] - op.f[12],
      this->f[13] - op.f[13],
      this->f[14] - op.f[14],
      this->f[15] - op.f[15]);
}

const Matrix44 Matrix44::operator*(float op) const
{
  Matrix44 ret;

  ret.f[0] = this->f[0] * op;
  ret.f[1] = this->f[1] * op;
  ret.f[2] = this->f[2] * op;
  ret.f[3] = this->f[3] * op;
  ret.f[4] = this->f[4] * op;
  ret.f[5] = this->f[5] * op;
  ret.f[6] = this->f[6] * op;
  ret.f[7] = this->f[7] * op;
  ret.f[8] = this->f[8] * op;
  ret.f[9] = this->f[9] * op;
  ret.f[10] = this->f[10] * op;
  ret.f[11] = this->f[11] * op;
  ret.f[12] = this->f[12] * op;
  ret.f[13] = this->f[13] * op;
  ret.f[14] = this->f[14] * op;
  ret.f[15] = this->f[15] * op;

  return ret;
}

const Matrix44 Matrix44::operator*(const Matrix44& op) const
{
  Matrix44 ret;

  ret.f[0] = this->f[0] * op.f[0] + this->f[4] * op.f[1] +
    this->f[8] * op.f[2] + this->f[12] * op.f[3];
  ret.f[4] = this->f[0] * op.f[4] + this->f[4] * op.f[5] +
    this->f[8] * op.f[6] + this->f[12] * op.f[7];
  ret.f[8] = this->f[0] * op.f[8] + this->f[4] * op.f[9] +
    this->f[8] * op.f[10] + this->f[12] * op.f[11];
  ret.f[12] = this->f[0] * op.f[12] + this->f[4] * op.f[13] +
    this->f[8] * op.f[14] + this->f[12] * op.f[15];

  ret.f[1] = this->f[1] * op.f[0] + this->f[5] * op.f[1] +
    this->f[9] * op.f[2] + this->f[13] * op.f[3];
  ret.f[5] = this->f[1] * op.f[4] + this->f[5] * op.f[5] +
    this->f[9] * op.f[6] + this->f[13] * op.f[7];
  ret.f[9] = this->f[1] * op.f[8] + this->f[5] * op.f[9] +
    this->f[9] * op.f[10] + this->f[13] * op.f[11];
  ret.f[13] = this->f[1] * op.f[12] + this->f[5] * op.f[13] +
    this->f[9] * op.f[14] + this->f[13] * op.f[15];

  ret.f[2] = this->f[2] * op.f[0] + this->f[6] * op.f[1] +
    this->f[10] * op.f[2] + this->f[14] * op.f[3];
  ret.f[6] = this->f[2] * op.f[4] + this->f[6] * op.f[5] +
    this->f[10] * op.f[6] + this->f[14] * op.f[7];
  ret.f[10] = this->f[2] * op.f[8] + this->f[6] * op.f[9] +
    this->f[10] * op.f[10] + this->f[14] * op.f[11];
  ret.f[14] = this->f[2] * op.f[12] + this->f[6] * op.f[13] +
    this->f[10] * op.f[14] + this->f[14] * op.f[15];

  ret.f[3] = this->f[3] * op.f[0] + this->f[7] * op.f[1] +
    this->f[11] * op.f[2] + this->f[15] * op.f[3];
  ret.f[7] = this->f[3] * op.f[4] + this->f[7] * op.f[5] +
    this->f[11] * op.f[6] + this->f[15] * op.f[7];
  ret.f[11] = this->f[3] * op.f[8] + this->f[7] * op.f[9] +
    this->f[11] * op.f[10] + this->f[15] * op.f[11];
  ret.f[15] = this->f[3] * op.f[12] + this->f[7] * op.f[13] +
    this->f[11] * op.f[14] + this->f[15] * op.f[15];

  return ret;
}

const Vector3 Matrix44::operator*(const Vector3& op) const
{
  Vector3 ret;

  ret.f[0] = this->f[0] * op.f[0] + this->f[4] * op.f[1] +
    this->f[8] * op.f[2] + this->f[12];
  ret.f[1] = this->f[1] * op.f[0] + this->f[5] * op.f[1] +
    this->f[9] * op.f[2] + this->f[13];
  ret.f[2] = this->f[2] * op.f[0] + this->f[6] * op.f[1] +
    this->f[10] * op.f[2] + this->f[14];

  return ret;
}

void Matrix44::operator+=(const Matrix44& op)
{
  this->f[0] += op.f[0];
  this->f[1] += op.f[1];
  this->f[2] += op.f[2];
  this->f[3] += op.f[3];
  this->f[4] += op.f[4];
  this->f[5] += op.f[5];
  this->f[6] += op.f[6];
  this->f[7] += op.f[7];
  this->f[8] += op.f[8];
  this->f[9] += op.f[9];
  this->f[10] += op.f[10];
  this->f[11] += op.f[11];
  this->f[12] += op.f[12];
  this->f[13] += op.f[13];
  this->f[14] += op.f[14];
  this->f[15] += op.f[15];
}

void Matrix44::operator-=(const Matrix44& op)
{
  this->f[0] -= op.f[0];
  this->f[1] -= op.f[1];
  this->f[2] -= op.f[2];
  this->f[3] -= op.f[3];
  this->f[4] -= op.f[4];
  this->f[5] -= op.f[5];
  this->f[6] -= op.f[6];
  this->f[7] -= op.f[7];
  this->f[8] -= op.f[8];
  this->f[9] -= op.f[9];
  this->f[10] -= op.f[10];
  this->f[11] -= op.f[11];
  this->f[12] -= op.f[12];
  this->f[13] -= op.f[13];
  this->f[14] -= op.f[14];
  this->f[15] -= op.f[15];
}

void Matrix44::operator*=(float op)
{
  this->f[0] *= op;
  this->f[1] *= op;
  this->f[2] *= op;
  this->f[3] *= op;
  this->f[4] *= op;
  this->f[5] *= op;
  this->f[6] *= op;
  this->f[7] *= op;
  this->f[8] *= op;
  this->f[9] *= op;
  this->f[10] *= op;
  this->f[11] *= op;
  this->f[12] *= op;
  this->f[13] *= op;
  this->f[14] *= op;
  this->f[15] *= op;
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif

