#ifndef effect_h
#define effect_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "enums.h"

#include "libfhi.h"

//############################################################################
// Define ####################################################################
//############################################################################

// Forward declaration.
class ParticleSystem;

//############################################################################
// Class #####################################################################
//############################################################################

/** One effect. Effects either happen instantaneously or they are copied to
 * the effect database for later execution. User does not have to know, he
 * executes the effects with the number of arguments desired, and they act
 * in any way neccessary.
 */
class Effect :
  public libfhi::Attributable,
  public libfhi::CanonicalDB<Effect>
{
  public:
    /** Defines if this should copy itself to the database. */
    static const uint8_t ATTR_TEMPORAL = 0x01;

    /** Default attributes. */
    static const uint8_t DEFAULT_ATTR = 0x00;

  private:
    /** Type of this effect. */
    EffectEnum type;

    /** Sparks to create on fade. */
    ParticleSystem *system;

    /** Minimum number of sparks. */
    int count_min;

    /** Maximum number of sparks. */
    int count_max;

    /** Minimum speed of sparks. */
    float speed_min;

    /** Maximum speed of sparks. */
    float speed_max;

    /** Lifetime for sparks. */
    int lifetime;

    /** Color of fade sparks. */
    libfhi::Color4 color;

  public:
    Effect();
    Effect(const char*);
    ~Effect();

    // Methods.
  public:
    void execute(const libfhi::Vector2*, const libfhi::Vector2*) const;
    void execute_now(const libfhi::Vector2*, const libfhi::Vector2*) const;

    // Inline
  public:
    inline int get_cnt() const;
    inline const libfhi::Color4& get_color() const;
    inline int get_lifetime() const;
    inline float get_speed() const;
    inline ParticleSystem* get_system() const;
    inline bool has_beam() const;
    inline bool has_temporal() const;
    inline bool has_thurster() const;
};

//############################################################################
// Effect inline #############################################################
//############################################################################

/** Get a randomized number describing how many particles to generate.
 * @return Random number.
 */
int Effect::get_cnt() const
{
  return rand() % (this->count_max - this->count_min + 1) + this->count_min;
}

/** Get initial color of particles generated by this.
 * @return Color as reference.
 */
const libfhi::Color4& Effect::get_color() const
{
  return this->color;
}

/** Return lifetime of particles generated by this.
 * @return Lifetime as int.
 */
int Effect::get_lifetime() const
{
  return this->lifetime;
}

/** Gets a random speed of this system. Do not use internally, can be
 * optimized.
 * @return Random speed within this.
 */
float Effect::get_speed() const
{
  return static_cast<float>(rand() % 1001) * 0.001f *
    (this->speed_max - this->speed_min) + this->speed_min;
}

/** Get the particle system associated with this.
 * @return Pointer to the system.
 */
ParticleSystem* Effect::get_system() const
{
  return this->system;
}

/** Tell if this effect lives.
 * @return True if yes, false if no.
 */
bool Effect::has_temporal() const
{
  return this->has_attr(ATTR_TEMPORAL);
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

