#include "encyclopediapage.h"

#include "encyclopedia.h"
#include "data.h"
#include "menu.h"

//############################################################################
// Constants #################################################################
//############################################################################

const float
  EncyclopediaPage::ENCYCLOPEDIA_X1 = 0.3f,
  EncyclopediaPage::ENCYCLOPEDIA_X2 = 1.0f,
  EncyclopediaPage::ENCYCLOPEDIA_Y1 = 0.2f,
  EncyclopediaPage::ENCYCLOPEDIA_Y2 = 1.0f;

//############################################################################
// Construction ##############################################################
//############################################################################

/* EncyclopediaPage constructor. Takes a filename as an argument and
 * constructs an encyclopedia page based on it, loading the appropriate data.
 * @param filename Input file.
 */
EncyclopediaPage::EncyclopediaPage(const char *filename)
  : normal_mesh(NULL), wireframe_mesh(NULL), content_header(NULL),
  content_image(NULL), content_specification(NULL), longest_radius(0.0f)
{
  libfhi::ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cerr << "Error: encyclopedia page \"" << filename <<
      "\" cannot be opened.\n";
  }

  while(filu.advance())
  {
    if(filu.has_id_arg("section", 2))
    {
      x1 = filu.get_float(0);
      y1 = filu.get_float(1);
    }
    else if(filu.has_id("name"))
    {
      this->name = filu.get_combined_string();
    }
    else if(filu.has_id_arg("image", 2))
    {
      content_image = Data::load_texture(filu.get_string(0).c_str());
      content_header = Data::load_texture(filu.get_string(1).c_str());
      type = ENCYCLOPEDIAPAGE_NORMAL;
    }
    else if(filu.has_id_arg("specification", 1))
    {
      //content_specification = libfhi::LoadImage(filu.get_combined_string());
      type = ENCYCLOPEDIAPAGE_SPECIFICATION;
    }
    else if(filu.has_id_arg("model", 1))
    {
      // Load the desired model.
      this->normal_mesh = Data::load_msh(filu.get_string(0).c_str());

      // Load a wireframe alteration of the mesh.
      libfhi::PreModel *pre = new libfhi::PreModel(
	  filu.get_string(0).c_str());
      pre->attr_or(libfhi::PreModel::ATTR_WIREFRAME);
      this->wireframe_mesh = pre->compile(NULL);
      
      // Set types and greatest ranges.
      type = ENCYCLOPEDIAPAGE_SPECIFICATION;
      longest_radius = this->normal_mesh->get_maxdist();
    }
    else if(filu.has_id("text"))
    {
      content_text = filu.get_combined_string();
    }
    else if(filu.has_id_arg("type", 1))
    {
      type = static_cast<EncyclopediaEnum>(filu.get_int(0));
    }
  }

  std::cout << "Loaded encyclopedia page \"" << filename << "\"\n";
}

/** Default destructor.
 */
EncyclopediaPage::~EncyclopediaPage()
{
  delete this->wireframe_mesh;
  delete this->content_header;
  delete this->content_image;
  delete this->content_specification;
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Display a model in a box.
 * @param vx1 Left x.
 * @param vy1 Top y.
 * @param vx2 Right x.
 * @param vy2 Down y.
 * @param mdl Model to render.
 * @param screen Target screen.
 */
void EncyclopediaPage::display_model(int vx1, int vy1, int vx2, int vy2,
    libfhi::Mesh *mesh, libfhi::Surface *screen)
{
  static libfhi::Camera camera;
  static libfhi::LightDirectional light;

  camera.set_pos(0.0f, 0.0f, this->longest_radius * 1.2f);

  screen->set_boundary(vx1, vy1, vx2, vy2, 1.0f, 0.1f, 100.0f);
  screen->set_camera(&camera);
  screen->set_light(&light);
  screen->select_3d();

  libfhi::PostModel *model = Encyclopedia::instance->get_model();
  model->adapt(mesh);
  libfhi::Surface::draw_model(model);
}

/** Draw this encyclopedia page on the screen. Utilizes full screen.
 * @param screen Screen to draw into.
 */
void EncyclopediaPage::draw(libfhi::Surface *screen)
{
  int view_x1,
      view_y1,
      view_x2 = screen->get_w() - 1,
      view_y2 = screen->get_h() - 1;

  // Clear the whole screen.
  screen->clear();
  screen->set_boundary(0, 0, view_x2, view_y2);
  screen->select_2d();

  // Draw outermost borders.
  libfhi::Surface::nc_rect(0, 0, view_x2, 1, Menu::get_color_border());
  libfhi::Surface::nc_rect(0, 2, 1, view_y2, Menu::get_color_border());
  libfhi::Surface::nc_rect(view_x2 - 1, 2, view_x2, view_y2,
      Menu::get_color_border());
  libfhi::Surface::nc_rect(2, view_y2 - 1, view_x2 - 2, view_y2,
      Menu::get_color_border());

  // Take the coordinates of the text box.
  view_x1 = lrintf(static_cast<float>(view_x2) * this->x1) + 2;
  view_y1 = lrintf(static_cast<float>(view_y2) * this->y1) + 2;
  view_x2 -= 2;
  view_y2 -= 2 + 2 + Menu::get_fontsize_desc() + 4;

  // Draw last separator borders.
  libfhi::Surface::nc_rect(2, view_y2 + 1, view_x2, view_y2 + 2,
      Menu::get_color_border());

  switch(type)
  {
    case ENCYCLOPEDIAPAGE_SPECIFICATION:
      {
	int middle_x = view_x2 / 2;

	// Draw borders.
	libfhi::Surface::nc_rect(2, view_y1 - 2, view_x2, view_y1 - 1,
	    Menu::get_color_border());
	libfhi::Surface::nc_rect(middle_x + 1, 2, middle_x + 2, view_y1 - 3,
	    Menu::get_color_border());

	// Model in top left box.
	this->display_model(2, 2, middle_x, view_y1 - 3, this->normal_mesh,
	    screen);
	this->display_model(middle_x + 3, 2, view_x2 - 2, view_y1 - 3,
	    this->wireframe_mesh, screen);
      }
      break;

    default: // ENCYCLOPEDIAPAGE_NORMAL
      {
	// Draw borders.
	libfhi::Surface::nc_rect(view_x1 - 2, 2, view_x1 - 1, view_y2,
	    Menu::get_color_border());
	libfhi::Surface::nc_rect(view_x1, view_y1 - 2, view_x2, view_y1 - 1,
	    Menu::get_color_border());

	if(this->content_image)
	{
	  libfhi::Surface::draw_texture(2, 2, view_x1 - 3, view_y2,
	      this->content_image);
	}

	if(this->content_header)
	{
	  libfhi::Surface::draw_texture(view_x1, 2, view_x2, view_y1 - 3,
	      this->content_header);
	}
      }
      break;
  }

  // Text box is universal.
  screen->set_boundary(0, 0, screen->get_w() - 1, screen->get_h() - 1);
  screen->select_2d();

  Menu::draw_text_box(view_x1 + 2, view_y1 + 2, view_x2 - 2, view_y2 - 2,
      this->content_text.c_str());
}

//############################################################################
// End #######################################################################
//############################################################################

