#ifndef game_h
#define game_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "constants.h"
#include "eventhandler.h"

#include HASHMAP_INCLUDE

//############################################################################
// Define ####################################################################
//############################################################################

// Forward declaration.
class EffectDB;
class Entity;
class EntityArchtype;
class ParticleSystem;
class StringInput;
class Terrain;
class TextOnPlayfield;
class Weapon;

//############################################################################
// Class #####################################################################
//############################################################################

/** Background light is a lighting schema to illuminate the back of the
 * playfield in a specific way.
 */
class Game :
  public EventHandler,
  public libfhi::Singleton<Game>
{
  protected:
    /** The frame number that tells which wrame this is. */
    static int frame_number;

    /** Ending frame. Compare against this, end the game upon reach. */
    static int frame_end;

  protected:
    /** Entity number. The id of next entity. */
    int entity_number;

    /** Entity table, for exists - not exists checks. */
    HASHMAP<size_t, bool> entity_table;

    /** Entity archtypes. */
    std::vector<EntityArchtype*> entity_archtypes;

    /** All the particle structures. */
    std::vector<ParticleSystem*> particlesystems;

    /** All the weapon structures. */
    std::vector<Weapon*> weapons;

    /** The effect database. */
    EffectDB *effectdb;

    /** Camera follow. */
    Entity* follow;

    /** The entities. */
    std::list<Entity*> entities;

    /** Entity destruction list. */
    std::list<Entity*> die_list;

    /** Text entries on the playfield. These are renderd after 3d. */
    std::list<TextOnPlayfield*> texts;

    /** Current viewpoort we're loking through. */
    libfhi::Vector2 viewport;

    /** One multimodel is player. */
    Entity *player;

    /** The camera viewing the playfield. */
    libfhi::Camera *camera;

    /** The light for viewing the objects. */
    libfhi::LightDirectional *light;

    /** Difficulties of entities. */
    HASHMAP<int, int> score_difficulty_by_entity;

    /** Entity groups by difficulty. */
    HASHMAP<int, std::vector<EntityEnum>*> score_entity_by_difficulty;

    /** The current score. */
    int score;

    /** Game ticks since last destruction (for score). */
    int destroy_tick;

    /** Game command key states. */
    int keys_now[COMMAND_TOTAL];

    /** Previous game command key states. */
    int keys_last[COMMAND_TOTAL];

    /** This will be used to collect score. Mostly NULL. */
    StringInput *string_input;

    /** True no game advancement (mostly false). */
    bool pause;

  public:
    Game();
    Game(const char*);
    virtual ~Game();

  protected:
    bool entity_delete(Entity*);
    bool entity_insert(Entity*, const libfhi::Vector2&, uint16_t, bool,
	float = 0.0f);
    Entity* entity_new(EntityArchtype*, int);
    Entity* entity_new(const char*, int);
    Entity* entity_new(EntityEnum, int);

    // Virtual inherited methods.
  public:
    virtual void draw(libfhi::Surface*);
    virtual void keydown(int);
    virtual void keyup(int);
    virtual void tick();

    // Entity handling.
  public:
    void add_effect(const char*);
    void add_entity_archtype(const char*);
    void add_particlesystem(const char*);
    void add_scoring_rule(EntityEnum, int);
    void add_text_on_playfield(const libfhi::Vector2&, const std::string&);
    void add_weapon(const char*);
    void draw_score(libfhi::Surface*) const;
    virtual void entity_delete(std::list<Entity*>::iterator, bool = false);
    int get_difficulty(EntityEnum);
    EntityEnum get_random_entity(int);
    int get_time_bonus_multiplier() const;

    // Inline methods
  public:
    inline bool entity_exists(Entity*);
    inline std::list<Entity*>* get_entity_list();
    inline ParticleSystem* get_particlesystem(ParticleEnum);
    inline const libfhi::Vector2& get_viewport() const;
    inline Weapon* get_weapon(WeaponEnum);

    // Static inline methods.
  public:
    static inline int get_frame_number();
    static inline bool inside_view_area(const libfhi::Vector2&, float);
    static inline void set_end_time();
};

//############################################################################
// Inline ####################################################################
//############################################################################

/** Check if given entity exists.
 * @param op Etity to check for.
 */
bool Game::entity_exists(Entity *op)
{
  size_t integer_representation = reinterpret_cast<size_t>(op);

  return (this->entity_table.find(integer_representation) !=
    this->entity_table.end());
}

/** Return the list of entities. This is used by ai's to determine the closest
 * enemies to attack.
 * @return Pointer to the entity list.
 */
std::list<Entity*>* Game::get_entity_list()
{
  return &(this->entities);
}

/** Get certain particle system data structure from the game.
 * @param enum describing the particle system index.
 * @return pointer to the weapon structure.
 */
ParticleSystem* Game::get_particlesystem(ParticleEnum op)
{
  return this->particlesystems[op];
}

/** Get current viewport.
 * @return Reference to the viewport vector.
 */
const libfhi::Vector2& Game::get_viewport() const
{
  return this->viewport;
}

/** Get certain weapon data structure from the game.
 * @param enum describing the weapon index.
 * @return pointer to the weapon structure.
 */
Weapon* Game::get_weapon(WeaponEnum op)
{
  return this->weapons[op];
}

//############################################################################
// Static inline #############################################################
//############################################################################

/** Get the current frame number.
 * @return The number.
 */
int Game::get_frame_number()
{
  return Game::frame_number;
}

/** Tell if position with a given maximum radius is still to be drawn.
 * @param pos Position.
 * @param radius Maximum radius thingy.
 * @return True if need to draw, false if not.
 */
bool Game::inside_view_area(const libfhi::Vector2 &pos, float radius)
{
  float border = VIEW_AREA_HALF + radius;

  if((pos.xf > border) || (pos.yf > border))
  {
    return false;
  }

  border = -border;

  return ((pos.xf >= border) || (pos.yf >= border));
}

/** Set time thus that the game will end.
 * Will not work on frame 0. Should not happen.
 */
void Game::set_end_time()
{
  Game::frame_end = Game::frame_number;
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

