#include "libfhi_point.h"
#include "libfhi_boundary.h"

#include <stdio.h>
#include <iostream>

namespace libfhi {

//############################################################################
// Position masks ############################################################
//############################################################################

/*
 * Mask setting functions work only on one dimension at a time. Rerun the
 * functions after each dimension loop.
 */

void Point::mask_x(const Boundary *boundary)
{
  if(boundary->above_x(pos.xf))
    mask = OUTSIDE_MAX;
  else if(boundary->below_x(pos.xf))
    mask = OUTSIDE_MIN;
  else
    mask = INSIDE;
}

void Point::mask_y(const Boundary *boundary)
{
  if(boundary->above_y(pos.yf))
    mask = OUTSIDE_MAX;
  else if(boundary->below_y(pos.yf))
    mask = OUTSIDE_MIN;
  else
    mask = INSIDE;
}

void Point::mask_z(const Boundary *boundary)
{
  if(boundary->above_z(pos.zf))
    mask = OUTSIDE_MAX;
  else if(boundary->below_z(pos.zf))
    mask = OUTSIDE_MIN;
  else
    mask = INSIDE;
}

//############################################################################
// Linear clipping ###########################################################
//############################################################################

/** 2d clipping, no colors, to X.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_x<Point2F>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_pos().clip2_x(lhs->get_pos(), rhs->get_pos(), value);
}

/** 2d clipping, no colors, to Y.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_y<Point2F>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_pos().clip2_y(lhs->get_pos(), rhs->get_pos(), value);
}

/** 2d clipping, with colors, to X.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_x<Point2G>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_col().clip3(lhs->get_col(), rhs->get_col(),
      dst->get_pos().clip2_x(lhs->get_pos(), rhs->get_pos(), value));
}

/** 2d clipping, with colors, to Y.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_y<Point2G>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_col().clip3(lhs->get_col(), rhs->get_col(),
      dst->get_pos().clip2_y(lhs->get_pos(), rhs->get_pos(), value));
}

/** 3d clipping, without colors, to X.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_x<Point3F>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_pos().clip3_x(lhs->get_pos(), rhs->get_pos(), value);
}

/** 3d clipping, without colors, to Y.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_y<Point3F>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_pos().clip3_y(lhs->get_pos(), rhs->get_pos(), value);
}

/** 3d clipping, without colors, to Z.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_z<Point3F>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_pos().clip3_z(lhs->get_pos(), rhs->get_pos(), value);
}

/** 3d clipping, with colors, to X.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_x<Point3G>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_col().clip3(lhs->get_col(), rhs->get_col(),
      dst->get_pos().clip3_x(lhs->get_pos(), rhs->get_pos(), value));
}

/** 3d clipping, with colors, to Y.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_y<Point3G>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_col().clip3(lhs->get_col(), rhs->get_col(),
      dst->get_pos().clip3_y(lhs->get_pos(), rhs->get_pos(), value));
}

/** 3d clipping, with colors, to Z.
 * @param dst Target point.
 * @param lhs Left clip point.
 * @param rhs Right clip point.
 * @param value Clip value.
 */
template <> void point_clip_z<Point3G>(Point *dst, const Point *lhs,
    const Point *rhs, float value)
{
  dst->get_col().clip3(lhs->get_col(), rhs->get_col(),
      dst->get_pos().clip3_z(lhs->get_pos(), rhs->get_pos(), value));
}

//############################################################################
// Setvintä ##################################################################
//############################################################################

/** Setvi, 2d, without colors.
 */
void Point2F::setvi()
{
  pos.round2();
}

/** Setvi, 2d, with colors.
 */
void Point2G::setvi()
{
  pos.round2();
  col.round3();
}

/** Setvi, 3d, without colors.
 */
void Point3F::setvi()
{
  pos.round3();
}

/** Setvi, 3d, with colors.
 */
void Point3G::setvi()
{
  pos.round3();
  col.round3();
}

//############################################################################
// Projektiot ################################################################
//############################################################################

/** Project this point according to given boundary, then calculate new Y mask.
 * @param boundary Pointer to the boundary to project with.
 */
void Point::project(const Boundary *boundary)
{
  pos.xf = boundary->xcenter - pos.xf / pos.zf * boundary->perspective; 
  pos.yf = pos.yf / pos.zf * boundary->perspective + boundary->ycenter;
  pos.zf = (pos.zf - boundary->zmax) * boundary->zmul;
  this->mask_y(boundary);
}

//############################################################################
// Debug #####################################################################
//############################################################################

#ifdef LIBFHI_DEBUG

std::ostream& Point2F::print(std::ostream &s) const
{
  return s << "<" << pos.xf << ", " << pos.yf << ">";
}

std::ostream& Point3F::print(std::ostream &s) const
{
  return s << "<" << pos.xf << ", " << pos.yf << ", " << pos.zf << ">";
}

std::ostream& Point2G::print(std::ostream &s) const
{
  return s << "<" << pos.xf << ", " << pos.yf << "> (" << col.rf << ", " <<
    col.gf << ", " << col.bf << ")";
}

std::ostream& Point3G::print(std::ostream &s) const
{
  return s << "<" << pos.xf << ", " << pos.yf << ", " << pos.zf << "> (" <<
    col.rf << ", " << col.gf << ", " << col.bf << ")";
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

}

