#include "libfhi_sdlsurface.h"
#include "libfhi_misc.h"

#include <iostream>

#include "SDL.h"

namespace libfhi {

//############################################################################
// Globaalit #################################################################
//############################################################################

const uint32_t sdlSurface::DEFAULT_SDL_FLAGS = SDL_SWSURFACE | SDL_ASYNCBLIT;

//############################################################################
// Konstruktio ###############################################################
//############################################################################

/** Default constructor.
 */
sdlSurface::sdlSurface()
{
  null();
}

/** Alternate constructor.
 * @param x Width.
 * @param y Height.
 * @param b Bit depth.
 * @param lf_flags Libfhi Flags for surface.
 * @param sdl_flags SDL flags for surface.
 */
sdlSurface::sdlSurface(int x, int y, int b, uint8_t lf_flags,
    uint32_t sdl_flags)
{
  null();
  reserve(x, y, b, lf_flags, sdl_flags);
}

/** Default destructor.
 */
sdlSurface::~sdlSurface()
{
  unreserve();
}

/** Emptying function.
 */
void sdlSurface::null()
{
  Surface::null();
  sdls = NULL;
}

//############################################################################
// Varaus ####################################################################
//############################################################################

/** Create an SDL display surface.
 * @param x [in] Width of the surface.
 * @param y [in] Heigth of the surface.
 * @param b [in] Depth of the surface in bits per pixel.
 * @param lf_flags [in] Libfhi flags of the surface.
 * @param sdl_flags [in] Additional SDL flags of the surface.
 * @return Memory (in bytes) reserved to the color and depth buffer.
 */
int sdlSurface::reserve(int x, int y, int b, uint8_t lf_flags,
    uint32_t sdl_flags)
{
  int rmask = 0, gmask = 0, bmask = 0, amask = 0;
  
  unreserve(); // Remove old values

  // Get new dimensions and check they're correct
  w = x;
  h = y;
  if((w <= 0) || (h <= 0))
  {
    std::cerr << "Error: Could not create surface: Invalid size " << w <<
      "x" << h << ".\n";
    unreserve();
    return 0;
  }

  // Get rest of the values.
  bpp = b;
  size_pixels = w * h;
  size_bitmap = size_pixels * (bpp / 8);

  // If bitmap reservation is on (default).
  if((lf_flags & FLAG_NOBITMAP) == 0)
  {
    // We create the surface ourselves, also set the status if explicitly told
    // to do so.
    if(lf_flags & FLAG_SET_MODE)
      sdls = SDL_SetVideoMode(w, h, bpp, sdl_flags);
    else
      sdls = SDL_CreateRGBSurface(sdl_flags, w, h, bpp, rmask, gmask,
	  bmask, amask);

    // Tarkastusta
    if(sdls == NULL)
    {
      std::cerr << "Error: Could not create surface: " << SDL_GetError() <<
	"\n";
      unreserve();
      return 0;
    }

    cbuf = static_cast<SDL_Surface*>(sdls)->pixels;
  }
 
  set_boundary();
  
#ifdef LIBFHI_DEBUG
  std::cout << "sdlSurface desired: ";
  print_sdls_flags(std::cout, sdl_flags);
  std::cout << "\nsdlSurface acquired: ";
  print_sdls_flags(std::cout, static_cast<SDL_Surface*>(sdls)->flags);
  std::cout << "\n";
#endif

  return size_bitmap + reserve_zbuffer(lf_flags);
}

/** Free all data reserved by this surface. Don't really know if this should
 * be a method at all instead of being in the destructor, but here it is.
 */
void sdlSurface::unreserve()
{
  if(sdls != NULL)
  {
    SDL_FreeSurface(static_cast<SDL_Surface*>(sdls));
    sdls = NULL;
    cbuf = NULL;
  }
  Surface::unreserve();
}

//############################################################################
// Blit ######################################################################
//############################################################################

/** Simple blit.
 * @param dx Destination left x.
 * @param dy Destination upper y.
 * @param s Destination image.
 */
void sdlSurface::blit(int dx, int dy, sdlSurface *s)
{
  SDL_Rect dst;
  dst.x = dx;
  dst.y = dy;
  SDL_BlitSurface(static_cast<SDL_Surface*>(sdls), NULL,
      static_cast<SDL_Surface*>(s->sdls), &dst);
}

/** Complex blit.
 * @param sx Source left x.
 * @param sy Source upper y.
 * @param sw Source width.
 * @param sh Sourch height.
 * @param dx Destination left x.
 * @param dy Destination upper y.
 * @param s Destination image.
 */
void sdlSurface::blit(int sx, int sy, int sw, int sh, int dx, int dy,
    sdlSurface *s)
{
  SDL_Rect src, dst;
  src.x = sx;
  src.y = sy;
  src.w = sw;
  src.h = sh;
  dst.x = dx;
  dst.y = dy;
  SDL_BlitSurface(static_cast<SDL_Surface*>(sdls), &src,
      static_cast<SDL_Surface*>(s->sdls), &dst);
}

//############################################################################
// Wrappityfunktiot ##########################################################
//############################################################################

/** Flip this SDL surface to the screen.
 */
void sdlSurface::flip()
{
  SDL_Flip(static_cast<SDL_Surface*>(sdls));
}

/** Lock the SDL surface. If not neccessary, does nothing.
 */
void sdlSurface::lock()
{
  SDL_LockSurface(static_cast<SDL_Surface*>(sdls));
}

/** Unlock the SDL surface. If not neccessary, does nothing.
 */
void sdlSurface::unlock()
{
  SDL_UnlockSurface(static_cast<SDL_Surface*>(sdls));
}

//############################################################################
// Sekalainen ################################################################
//############################################################################

/*
 * Adoptoi SDL-pinta
 */

/** Adapt a SDL_Surface.
 * @param op [in] The surface to adapt.
 */
void sdlSurface::adapt(void *op)
{
  SDL_Surface *surf = static_cast<SDL_Surface*>(op);

  if(surf == NULL)
    return;

  sdls = surf;
  cbuf = surf->pixels;
  w = surf->w;
  h = surf->h;
  bpp = surf->format->BitsPerPixel;
  set_boundary();
} 

//############################################################################
// Factory-funktiot ##########################################################
//############################################################################

/** Set the given graphics mode.
 * @param x Width of the mode.
 * @param y Height of the mode.
 * @param bpp Bits per pixel in the mode.
 * @param lf_flags Libfhi surface flags if not default.
 * @param sdl_flags SDL surface flags if not default.
 * @return New surface.
 */
sdlSurface* set_mode(int x, int y, int bpp, uint8_t lf_flags,
    uint32_t sdl_flags)
{
  return sdlsurface_new(x, y, bpp, lf_flags | Surface::FLAG_SET_MODE,
      sdl_flags);
}

/** Only reserve non-mode setting surfaces through this function.
 * @param x Width of the mode.
 * @param y Height of the mode.
 * @param bpp Bits per pixel in the mode.
 * @param lf_flags Libfhi surface flags if not default.
 * @param sdl_flags SDL surface flags if not default.
 * @return New surface.
 */
sdlSurface* sdlsurface_new(int x, int y, int bpp, uint8_t lf_flags,
    uint32_t sdl_flags)
{
  sdlSurface *ret = NULL;

  switch(bpp)
  {
    case 8:
    case 16:
    case 32:
      ret = new sdlSurface(x, y, bpp, lf_flags, sdl_flags);
      break;
    default:
      std::cerr << "Error: Could not reserve sdlSurface.\n";
      return NULL;
  }

  return ret;
}

//###########################################################################
// Debug ####################################################################
//###########################################################################

#ifdef LIBFHI_DEBUG

std::ostream& sdlSurface::print_sdls_flags(std::ostream &s, uint32_t flags)
{
  s << "(";

  // HW/SW
  if(flags & SDL_HWSURFACE)
  {
    s << "HW";
  }
  else
  {
    s << "SW";
  }

  // Other flags.
  if(flags & SDL_ASYNCBLIT)
  {
    s << " | ASYNCBLIT";
  }
  if(flags & SDL_ANYFORMAT)
  {
    s << " | ANYFORMAT";
  }
  if(flags & SDL_HWPALETTE)
  {
    s << " | HWPALETTE";
  }
  if(flags & SDL_DOUBLEBUF)
  {
    s << " | DOUBLEBUF";
  }
  if(flags & SDL_FULLSCREEN)
  {
    s << " | FULLSCREEN";
  }
  if(flags & SDL_OPENGL)
  {
    s << " | OPENGL";
  }
  if(flags & SDL_OPENGLBLIT)
  {
    s << " | OPENGLBLIT";
  }
  if(flags & SDL_HWACCEL)
  {
    s << " | HWACCEL";
  }
  if(flags & SDL_SRCCOLORKEY)
  {
    s << " | SRCCOLORKEY";
  }

  return s << ")";
}

#endif

//###########################################################################
// Loppu ####################################################################
//###########################################################################

}

