using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Text;
using System.Windows.Forms;

namespace StarifficEditor
{
    public partial class DoExportForm : Form
    {
        private ExportOptions m_options;
        private String m_export_path;
        private Meshes m_meshes;

        public DoExportForm()
        {
            InitializeComponent();
        }

        public DoExportForm(ExportOptions options, String export_path, Meshes meshes) : this()
        {
            m_options = options;
            m_export_path = export_path;
            m_meshes = meshes;
        }

        private void backgroundWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            BackgroundWorker bw = sender as BackgroundWorker;

            // Clean the export folder
            bw.ReportProgress(0, " * Clearing export folder..");
            foreach (String fname in Directory.GetFiles(m_export_path))
            {
                try
                {
                    bw.ReportProgress(0, "Deleting: "+fname);
                    File.Delete(fname);
                }
                catch (Exception)
                {
                    bw.ReportProgress(0, "(error removing file!)");
                }
            }

            // Export the meshes
            bw.ReportProgress(0, " * Exporting meshes..");

            // Export the levels
            LevelManifest manifest = new LevelManifest();

            bw.ReportProgress(0, " * Exporting levels..");
            for (int i = 0; i < m_options.levels.Count; ++i)
            {
                String level_name = m_options.levels[i];
                String dest_name = "" + (i + 1);
                String src = Path.Combine(m_options.LevelPath, level_name);
                String dest = Path.Combine(m_export_path, dest_name);

                Level level = null;
                try
                {
                    bw.ReportProgress(0, "Converting level: " + level_name);

                    // Load the level
                    level = new Level();
                    if (!level.Load(src))
                    {
                        bw.ReportProgress(0, "(error loading!)");
                        continue;
                    }

                    // Export the level
                    if (!level.Export(dest))
                    {
                        bw.ReportProgress(0, "(error exporting)");
                        continue;
                    }
                }
                catch (Exception)
                {
                    bw.ReportProgress(0, "(error exporting; exception thrown!)");
                    continue;
                }

                // Add to manifest
                manifest.AddLevel(level, "/" + dest_name);
            }

            // Export the manifest
            bw.ReportProgress(0, " * Exporting the level manifest");
            if (!manifest.Export(Path.Combine(m_export_path, "mft")))
                bw.ReportProgress(0, "(export failed!)");

            // Export the meshes
            bw.ReportProgress(0, " * Exporting the meshes");
            if (!m_meshes.Export(Path.Combine(m_export_path, "msh")))
                bw.ReportProgress(0, "(export failed!)");
        }

        private void backgroundWorker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            String msg = e.UserState as String;
            if (msg == null)
                return;

            listBoxMessages.Items.Add(msg);
            listBoxMessages.SelectedIndex = listBoxMessages.Items.Count - 1;
        }

        private void backgroundWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            listBoxMessages.Items.Add("Done!");
            listBoxMessages.SelectedIndex = listBoxMessages.Items.Count - 1;
            buttonOk.Enabled = true;
        }

        private void buttonOk_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void DoExportForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (buttonOk.Enabled == false)
            {
                e.Cancel = true;
            }
        }

        private void DoExportForm_Load(object sender, EventArgs e)
        {
            try
            {
                backgroundWorker.RunWorkerAsync();
            }
            catch (Exception ex)
            {
                listBoxMessages.Items.Add("ERROR STARTING WORK: " + ex);
                buttonOk.Enabled = true;
            }
        }
    }
}