#include <SDL.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "sdl.h"
#include "timer.h"
#include "wiimote.h"

GLuint particle3d;

void init_fluid3d()
{
  load_texture("particle.bmp", &particle3d);
  
  glEnable( GL_TEXTURE_2D );
  glShadeModel( GL_SMOOTH );

  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
  glDisable(GL_DEPTH_TEST);
  

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  glEnable(GL_TEXTURE_2D);
  

/*
  glClearDepth(1.0f);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
*/
}

void draw_particle3d(float x, float y, float z)
{
  float size = 0.5;
  glBegin(GL_TRIANGLE_STRIP);
  glTexCoord2d(1, 1);
  glVertex3f(x + size, y + size, z);
  glTexCoord2d(0, 1);
  glVertex3f(x - size, y + size, z);
  glTexCoord2d(1, 0);
  glVertex3f(x + size, y - size, z);
  glTexCoord2d(0, 0);
  glVertex3f(x - size, y - size, z);
  glEnd();
}

float rotation3d = 45.0;

#define IX(i,j,k) ((i)+(N+2)*(j)+(N+2)*(N+2)*(k))

void set_bnd3d( int N, int b, float * x )
{
  int i;
  for ( i=1 ; i<=N ; i++ )
  {
    x[IX(  0,   i,   0)] = 0; //b==1 ? -x[IX(1, i, 1)] : x[IX(1, i, 1)]; //1
    x[IX(N+1,   i,   0)] = 0; //b==1 ? -x[IX(N, i, 1)] : x[IX(N, i, 1)]; //2
    x[IX(  i,   0,   0)] = 0; //b==2 ? -x[IX(i, 1, 1)] : x[IX(i, 1, 1)]; //3
    x[IX(  i, N+1,   0)] = 0; //b==2 ? -x[IX(i, N, 1)] : x[IX(i, N, 1)]; //4
    x[IX(  0,   0,   i)] = 0; //b==3 ? -x[IX(1, 1, i)] : x[IX(1, 1, i)]; //5
    x[IX(N+1,   0,   i)] = 0; //b==3 ? -x[IX(N, 1, i)] : x[IX(N, 1, i)]; //6
    x[IX(  0, N+1,   i)] = 0; //b==3 ? -x[IX(1, N, i)] : x[IX(1, N, i)]; //7
    x[IX(N+1, N+1,   i)] = 0; //b==3 ? -x[IX(N, N, i)] : x[IX(N, N, i)]; //8
  
    x[IX(  0,   i, N+1)] = 0; //b==1 ? -x[IX(1, i, N)] : x[IX(1, i, N)]; //9
    x[IX(N+1,   i, N+1)] = 0; //b==1 ? -x[IX(N, i, N)] : x[IX(N, i, N)]; //10
    x[IX(  i,   0, N+1)] = 0; //b==2 ? -x[IX(i, 1, N)] : x[IX(i, 1, N)]; //11
    x[IX(  i, N+1, N+1)] = 0; //b==2 ? -x[IX(i, N, N)] : x[IX(i, N, N)]; //12
  }
  x[IX(0 ,0, 0 )]   = 0.0;
  x[IX(0 ,N+1, 0)]  = 0.0;
  x[IX(N+1,0, 0)]  = 0.0;
  x[IX(N+1,N+1, 0)] = 0.0;
  
  x[IX(0 ,0, N+1 )]   = 0.0;
  x[IX(0 ,N+1, N+1)]  = 0.0;
  x[IX(N+1,0, N+1)]  = 0.0;
  x[IX(N+1,N+1, N+1)] = 0.0;
}

void project3d ( int N, float * u, float * v, float * w, float * p, float * div, float *diw )
{
    int i, j, k, l;
    float h;
    h = 1.0/N;
    for ( i=1 ; i<=N ; i++ ) {
        for ( j=1 ; j<=N ; j++ ) {
          for ( k=1 ; k<=N ; k++ ) {
            div[IX(i,j,k)] = -(1.0/3.0)*h*(u[IX(i+1,j,k)]-u[IX(i-1,j,k)]+
                                   v[IX(i,j+1,k)]-v[IX(i,j-1,k)]+
                                   w[IX(i,j,k+1)]-w[IX(i,j,k-1)]);
            p[IX(i,j,k)] = 0;
          }
        }
    }
    set_bnd3d ( N, 0, div ); set_bnd3d ( N, 0, p );
    for ( l=0 ; l<20 ; l++ ) {
        for ( i=1 ; i<=N ; i++ ) {
            for ( j=1 ; j<=N ; j++ ) {
              for ( k=1 ; k<=N ; k++ ) {
                p[IX(i,j,k)] = (div[IX(i,j,k)]+p[IX(i-1,j,k)]+p[IX(i+1,j,k)]+
                                            p[IX(i,j-1,k)]+p[IX(i,j+1,k)]+
                                            p[IX(i,j,k-1)]+p[IX(i,j,k+1)])/6;
              }
            }
        }
        set_bnd3d ( N, 0, p );
    }
    for ( i=1 ; i<=N ; i++ ) {
        for ( j=1 ; j<=N ; j++ ) {
          for ( k=1 ; k<=N ; k++ ) {
            u[IX(i,j,k)] -= 0.5*(p[IX(i+1,j,k)]-p[IX(i-1,j,k)])/h;
            v[IX(i,j,k)] -= 0.5*(p[IX(i,j+1,k)]-p[IX(i,j-1,k)])/h;
            w[IX(i,j,k)] -= 0.5*(p[IX(i,j,k+1)]-p[IX(i,j,k-1)])/h;
          }
        }
    }
    set_bnd3d ( N, 1, u ); set_bnd3d ( N, 2, v ); set_bnd3d ( N, 3, w );
}

void add_source3d(int N, float * x, float * s, float dt)
{
  int i, size=(N+2)*(N+2)*(N+2);
  for ( i=0 ; i<size ; i++ ) x[i] += dt*s[i];
}

void diffuse3d ( int N, int b, float * x, float * x0, float diff, float dt )
{
    int i, j, k, l;
    float a=dt*diff*N*N;
    for ( l=0 ; l<20 ; l++ ) {
        for ( i=1 ; i<=N ; i++ ) {
            for ( j=1 ; j<=N ; j++ ) {
              for ( k=1 ; k<=N ; k++ ) {
                x[IX(i,j,k)] = (x0[IX(i,j,k)] + a*(x[IX(i-1,j,k)]+x[IX(i+1,j,k)]+
                                               x[IX(i,j-1,k)]+x[IX(i,j+1,k)]+
                                               x[IX(i,j,k-1)]+x[IX(i,j,k+1)]))/(1+6*a);
              }
            }
        }
        set_bnd3d ( N, b, x );
    }
}

void advect3d ( int N, int b, float * d, float * d0, float * u, float * v, float * w, float dt )
{
    int i, j, k, i0, j0, k0, i1, j1, k1;
    float x, y, z, s0, t0, r0, s1, t1, r1, dt0;
    dt0 = dt*N;
    for ( i=1 ; i<=N ; i++ ) {
        for ( j=1 ; j<=N ; j++ ) {
          for ( k=1 ; k<=N ; k++ ) {
            x = i-dt0*u[IX(i,j,k)];
            y = j -dt0*v[IX(i,j,k)];
            z = k -dt0*w[IX(i,j,k)];
            if (x<0.5) x=0.5; if (x>N+0.5) x=N+ 0.5; i0=(int)x; i1=i0+ 1;
            if (y<0.5) y=0.5; if (y>N+0.5) y=N+ 0.5; j0=(int)y; j1=j0+1;
            if (z<0.5) z=0.5; if (z>N+0.5) z=N+ 0.5; k0=(int)z; k1=k0+1;
            s1 = x-i0; s0 = 1-s1;
            t1 = y-j0; t0 = 1-t1;
            r1 = z-k0; r0 = 1-r1;
            d[IX(i,j,k)] = r0*(s0*(t0*d0[IX(i0,j0,k0)]+t1*d0[IX(i0,j1,k0)])+
                          s1*(t0*d0[IX(i1,j0,k0)]+t1*d0[IX(i1,j1,k0)]))+
                          r1*(s0*(t0*d0[IX(i0,j0,k1)]+t1*d0[IX(i0,j1,k1)])+
                          s1*(t0*d0[IX(i1,j0,k1)]+t1*d0[IX(i1,j1,k1)]));
        }
      }
  }
  set_bnd3d ( N, b, d );
}

#define SWAP(x0,x) {float *tmp=x0;x0=x;x=tmp;}

void dens_step3d ( int N, float * x, float * x0, float * u, float * v, float * w, float diff,
    float dt )
{
    add_source3d ( N, x, x0, dt );
    SWAP ( x0, x ); diffuse3d ( N, 0, x, x0, diff, dt );
    SWAP ( x0, x ); advect3d ( N, 0, x, x0, u, v, w, dt );
}

void vel_step3d ( int N, float * u, float * v, float * w, float * u0, float * v0, float * w0,
           float visc, float dt )
{
    add_source3d ( N, u, u0, dt );
    add_source3d ( N, v, v0, dt );
    add_source3d ( N, w, w0, dt );
    SWAP ( u0, u ); diffuse3d ( N, 1, u, u0, visc, dt);
    SWAP ( v0, v ); diffuse3d ( N, 2, v, v0, visc, dt);
    SWAP ( w0, w ); diffuse3d ( N, 3, w, w0, visc, dt);
    project3d ( N, u, v, w, u0, v0, w0 );
    SWAP ( u0, u );
    SWAP ( v0, v );
    SWAP ( w0, w );
    advect3d ( N, 1, u, u0, u0, v0, w0, dt );
    advect3d ( N, 2, v, v0, u0, v0, w0, dt );
    advect3d ( N, 3, w, w0, u0, v0, w0, dt );
    project3d ( N, u, v, w, u0, v0, w0 );
}

#define GRID_N 16
#define SIZE ((GRID_N+2)*(GRID_N+2)*(GRID_N+2))
static float u[SIZE], v[SIZE], w[SIZE], u_prev[SIZE], v_prev[SIZE], w_prev[SIZE];
static float dens[SIZE], dens_prev[SIZE];

void get_from_UI3d (int N, float * d, float * u, float * v, float * w )
{
  int i, j, k, size = (N+2)*(N+2)*(N+2);

  for ( i=0 ; i<size ; i++ ) {
          u[i] = v[i] = d[i] = 0.0f;
  }
  
  i = N/2;
  j = N/2;
  k = N/2;
  d[IX(i, j, k)] = (wiimote1->gforce.x * wiimote1->gforce.x + wiimote1->gforce.y * wiimote1->gforce.y + wiimote1->gforce.z * wiimote1->gforce.z) * 5.0;
  u[IX(i, j, k)] = wiimote1->gforce.x * 5.0;
  v[IX(i, j, k)] = wiimote1->gforce.y * 5.0;
  w[IX(i, j, k)] = wiimote1->gforce.z * 5.0;
  
  /*
  if (IS_PRESSED(wiimote1, WIIMOTE_BUTTON_B))
  {
    i = N/2;
    j = N/2;
    k = N/2;
    d[IX(i, j, k)] = 10.0;
    u[IX(i, j, k)] = 1.0;
    v[IX(i, j, k)] = 0.0;
    w[IX(i, j, k)] = 0.0;
  }
  
  if (IS_PRESSED(wiimote1, WIIMOTE_BUTTON_A))
  {
    i = N/2;
    j = N/2;
    k = N/2;
    d[IX(i, j, k)] = 10.0;
    u[IX(i, j, k)] = 0.0;
    v[IX(i, j, k)] = 1.0;
    w[IX(i, j, k)] = 0.0;
  }
  
  if (IS_PRESSED(wiimote1, WIIMOTE_BUTTON_DOWN))
  {
    i = N/2;
    j = N/2;
    k = N/2;
    d[IX(i, j, k)] = 10.0;
    u[IX(i, j, k)] = 0.0;
    v[IX(i, j, k)] = 0.0;
    w[IX(i, j, k)] = 1.0;
  }
  */
}

void update_dens3d(float dt)
{
  float visc = 0.0;
  float diff = 0.0;
  int N = GRID_N;
  get_from_UI3d ( N, dens_prev, u_prev, v_prev, w_prev );
  vel_step3d ( N, u, v, w, u_prev, v_prev, w_prev, visc, dt );
  dens_step3d ( N, dens, dens_prev, u, v, w, diff, dt );
}

void draw_dens3d(int N, float *dens)
{
  float x = 0, y = 0, z = 30-50;
  float r = 1.0f, g = 1.0f, b = 1.0f, a = 1.0f;
  int i, j, k;
  /*
  if (wiimote_ticks_z)
    tick_counter = 100;
  
  if (tick_counter > 0)
    r = 0.0;
  
  tick_counter -= frame_ticks;
  */
  
  glLoadIdentity();
  glTranslatef(0, 0, -20);
  glRotatef(rotation3d, 0, 1, 0);
  glTranslatef(0, 0, 20);
  //rotation3d += frame_ticks * 0.01;
  //glTranslatef(-5, 0, 0);
  for (i=1; i<=N; i++)
    for (j=1; j<=N; j++)
      for (k=1; k<=N; k++)
      {
        x = (((float)i / GRID_N) * 2.0 - 1.0) * 5;
        y = (((float)j / GRID_N) * 2.0 - 1.0) * 5;
        z = (((float)k / GRID_N) * 2.0 - 1.0) * 5;
        a = dens[IX(i,j,k)];
        if (a < 0) a = 0.0;
        if (a > 1.0) a = 1.0;
        //printf("%g\n", a);
        
        glColor4f(r, g, b, a);
        draw_particle3d(x, y, -20+z);
      }
}


void draw_fluid3d()
{
  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  
  update_dens3d(frame_ticks / 100.0);
  draw_dens3d(GRID_N, dens);
  
  if (1)
  {
    glLoadIdentity();
    glColor4f(1.0, 1.0, 1.0, 1.0);
    draw_particle3d(-8, wiimote_x, -20);
    draw_particle3d(-10, wiimote_z, -20);
  }
  
  //printf("%f %f\n", wiimote_z, wiimote_amp_z);
  /*
  glLoadIdentity();
  glTranslatef(0, 0, -10);
  
  glBegin(GL_TRIANGLE_STRIP);
  glColor4f(1.0, 0.0, 0.0, 1.0);
  glVertex3f(1.0, 0.0, 0.0);
  
  glColor4f(0.0, 0.0, 0.0, 1.0);
  glVertex3f(0.0, 0.0, 0.0);
  
  glColor4f(1.0, 1.0, 1.0, 1.0);
  glVertex3f(1.0, 1.0, 0.0);
  
  glColor4f(0.0, 1.0, 0.0, 1.0);
  glVertex3f(0.0, 1.0, 0.0);
  glEnd();
  */
}
