import pygame
from pygame import *
from settings import *
from platform import *
from consumable import *
from item import *
from unlockable import *
from anim import *
from bullet import *
from room import *
from shadow import *
import random

#height/width of player
w = 24*sizeMultiplier
h = 48*sizeMultiplier

dw = 24*sizeMultiplier
dh = 32*sizeMultiplier


class Player(pygame.sprite.Sprite):
    """
    This class controls the player
    """

    def __init__(self, x, y, startingRoom):
        pygame.sprite.Sprite.__init__(self)
        self.xvel = 0
        self.yvel = 0
        
        #player states
        self.onGround = False
        self.faceright = True
        self.holdingJump = False
        self.hit = False
        self.holdingTime = 0
        self.duck = False
        self.holdingItem = None
        self.attack = False
        self.walking = False
        self.double_down = False
        self.inAir = False
        self.inPortal = False

        #load animations
        self.walk = Animation("charlie_trombone_walk_final", "#000000")
        self.walk_attack = Animation("charlie_walk_trombone_shoot", "#000000")
        self.idle_attack = Animation("charlie_trombone_idle_shoot", "#000000")

        self.duck_walk = Animation("charlie_trombone_duck_walk", "#000000")
        self.duck_walk_attack = Animation("charlie_trombone_duck_shoot_walk", "#000000")
        self.duck_idle_attack = Animation("charlie_trombone_duck_idle_shoot", "#000000")

        self.shootJumpUp = Animation("charlie_jump_up_shoot", "#000000")
        self.shootJumpDown = Animation("charlie_jump_down_shoot", "#000000")

        #load idle images
        self.idleLeft = loadSprite("charlie_trombone_idle")
        self.idleRight = pygame.transform.flip(self.idleLeft, True, False)

        self.idleLeftJump = loadSprite("charlie_trombone_idle_jump")
        self.idleRightJump = pygame.transform.flip(self.idleLeftJump, True, False)

        self.duck_idleLeft = loadSprite("charlie_trombone_duck_idle")
        self.duck_idleRight = pygame.transform.flip(self.duck_idleLeft, True, False)

        self.idleLeftJumpUp = loadSprite("charlie_jump_up_idle")
        self.idleRightJumpUp = pygame.transform.flip(self.idleLeftJumpUp, True, False)

        self.idleLeftJumpDown = loadSprite("charlie_jump_down_idle")
        self.idleRightJumpDown = pygame.transform.flip(self.idleLeftJumpDown, True, False)


        #set default image
        self.image = self.idleRight

        #shadow
        self.shadow = Shadow("cons_shadow")

        #setup collision boxes
        self.rect = Rect(0, 0, w, h)
        self.standrect = self.rect
        self.sitrect = Rect(0, 0, dw, dh)

        self.rect.left += 1 * size
        self.rect.top += 11 * size

        #player stats
        self.lives = 4      #number of heart containers - max 16
        self.life = 4       #how much life is left
        self.keys = 0       #number of keys
        self.money = 0      #amount of money

        #player weapon stats
        self.damage = 1
        self.speed = 10
        self.rate = 20
        self.rateCount = 0
        self.range = 400

        #items that can be taken
        self.items = [subItem for subItem in Item.__subclasses__()] #get all subclasses of Item
        random.shuffle(self.items)

        #dirty rect
        self.oldPlayerDirty = Rect(0,0,0,0)

    def update(self, crouch, left, right, attack, jump, double_down, room):
        """
        Compute the next state and position of the player based on input from keyboard.
        """

        self.double_down = double_down

        #player is holding item above head, freeze controls
        if self.holdingItem:
            if self.holdingItem.stillHolding(room):
                crouch = left = right = attack = jump = double_down = False
            else:
                self.holdingItem = None

        #player got hit -> temporary invincible
        if self.hit and self.hitCounter < 100:
            self.hitCounter += 1
        else:
            self.hit = False
            self.hitCounter = 0

        #make player duck
        if crouch and not self.duck:
            self.sitrect.topleft = self.rect.topleft
            self.rect = self.sitrect
            self.rect.top += h-dh
            self.duck = True

        #try standing up
        elif not crouch and self.duck:

            #assume player can stand up
            self.standrect.topleft = self.rect.topleft
            self.rect = self.standrect
            self.rect.top -= h-dh
            self.duck = False

            #check if standing up works
            for p in room.platforms:
                if sprite.collide_rect(self, p):
                    self.sitrect.topleft = self.rect.topleft
                    self.rect = self.sitrect
                    self.rect.top += h-dh
                    self.duck = True
                    break

        self.crouch = crouch

        #jump
        if jump:
            if self.onGround:
                self.holdingJump = True

            if self.holdingJump:
                self.holdingTime += 1

                self.yvel = -10 - self.holdingTime

                if self.holdingTime > 7:
                    self.yvel = -17

                if self.holdingTime > 13:
                    self.holdingJump = False
        else:
            self.holdingJump = False
            self.holdingTime = 0


        #move left
        if left:
            self.xvel -= 0.25*sizeMultiplier

            if self.xvel < -5*sizeMultiplier:
                self.xvel = -5*sizeMultiplier

            self.faceright = False

        #move right
        if right:
            self.xvel += 0.25*sizeMultiplier

            if self.xvel > 5*sizeMultiplier:
                self.xvel = 5*sizeMultiplier

            self.faceright = True

        #player is falling, apply gravitation
        if not self.onGround:
            self.yvel += 0.4*sizeMultiplier

            if self.yvel > 25*sizeMultiplier:
                self.yvel = 25*sizeMultiplier

        #Emulate friction
        if not(left or right):
            if self.xvel > 0:
                self.xvel -= 0.125*sizeMultiplier

            if self.xvel < 0:
                self.xvel += 0.125*sizeMultiplier

        #check x-axis collisions
        self.rect.left += self.xvel
        self.collideXvel(self.xvel, room)

        #check y-axis collisions
        self.rect.top += self.yvel
        self.onGround = False
        self.collideYvel(self.yvel, room)

        self.dynamicCollide(room)
                
        #attack
        if attack:
            self.attack = True

            #control firerate
            if self.rateCount > self.rate:
                self.rateCount = 0

            #create bullet at player position
            if self.rateCount == 0:
                #find out where bullets should spawn
                if self.faceright:
                    x,y = self.rect.topright
                    direction = 0
                else:
                    x,y = self.rect.topleft
                    direction = 180

                room.bullets.append(PlayerBullet(x, y + 5, math.radians(direction), self))

        self.rateCount += 1

        #update shadow
        self.shadow.update(self, room.platforms + room.doors + room.steps)

    def draw(self, screen, camera):
        """
        Draw player sprite

        Returns dirty rectangle
        """

        #draw shadow
        screen.blit(self.shadow.image, camera.apply(self.shadow))

        #check if we are walking
        if self.xvel == 0:
            self.walking = False
        else:
            self.walking = True

        #check if player is in air
        if self.yvel < 0: 
            self.inAir = True
        else:
            self.inAir = False

        #Find out which frame of which animation to show
        if self.walking:
            if self.duck:
                #sync walking
                self.duck_walk.advance()
                self.duck_walk_attack.advance()

                if self.attack:
                    self.image = self.duck_walk_attack.current(self.faceright)
                    self.attack = False

                else:
                    self.image = self.duck_walk.current(self.faceright)
            else:
                #sync walking
                self.walk.advance()
                self.walk_attack.advance()

                if self.attack:
                    if self.yvel < -0.1 or self.yvel > 0.8:
                        if self.yvel > 0:
                            self.image = self.shootJumpDown.next(self.faceright)
                        else:
                            self.image = self.shootJumpUp.next(self.faceright)

                    else:
                        self.image = self.walk_attack.current(self.faceright)
                    
                    self.attack = False

                else:
                    if self.faceright:
                        if self.yvel < -0.1 or self.yvel > 0.8:
                            if self.yvel > 0:
                                self.image = self.idleRightJumpDown
                            else:
                                self.image = self.idleRightJumpUp
                        
                        else:
                            self.image = self.walk.current(self.faceright)

                    else:
                        if self.yvel < -0.1 or self.yvel > 0.8:
                            if self.yvel > 0:
                                self.image = self.idleLeftJumpDown
                            else:
                                self.image = self.idleLeftJumpUp
                        else:
                            self.image = self.walk.current(self.faceright)

        #idle graphics
        else:
            if self.duck:
                self.duck_walk.reset()
                self.duck_walk_attack.reset()

                if self.attack:
                    self.image = self.duck_idle_attack.next(self.faceright)
                    self.attack = False

                else:
                    if self.faceright:
                        self.image = self.duck_idleRight
                    else:
                        self.image = self.duck_idleLeft
            else:
                self.walk.reset()
                self.walk_attack.reset()

                if self.attack:
                    if self.yvel < -0.1 or self.yvel > 0.8:
                        if self.yvel > 0:
                            self.image = self.shootJumpDown.next(self.faceright)
                        else:
                            self.image = self.shootJumpUp.next(self.faceright)
                    else:
                        self.image = self.idle_attack.next(self.faceright)

                    self.attack = False

                else:
                    if self.faceright:
                        if self.yvel < -0.1 or self.yvel > 0.8:
                            if self.yvel > 0:
                                self.image = self.idleRightJumpDown
                            else:
                                self.image = self.idleRightJumpUp
                        
                        else:
                            self.image = self.idleRight
                    else:
                        if self.yvel < -0.1 or self.yvel > 0.8:
                            if self.yvel > 0:
                                self.image = self.idleLeftJumpDown
                            else:
                                self.image = self.idleLeftJumpUp
                        
                        else:
                            self.image = self.idleLeft

        #draw player
        nx, ny = camera.apply(self).topleft
        if self.faceright:
            nx -= 13*sizeMultiplier
        else:
            nx -= 31*sizeMultiplier

        screen.blit(self.image, (nx,ny))

        #handle dirty rectangles
        dirty = []

        if dirtOpt:
            playerDirty = Rect(nx,ny,64*2,48*2)
            dirty = [self.oldPlayerDirty, playerDirty]
            self.oldPlayerDirty = playerDirty

        return dirty


    def collideXvel(self, xvel, room):
        """
        Check collision with static objects on x axis
        """
        if room.locked:
            solids = room.platforms + room.doors
        else:
            solids = room.platforms

        solids = solids + room.steps

        for p in solids:
            if sprite.collide_rect(self, p):
                if not isinstance(p,Step):

                    #collision right
                    if xvel > 0:
                        self.rect.right = p.rect.left
                        self.xvel -= 0.25
                    
                    #collision left
                    if xvel < 0:
                        self.rect.left = p.rect.right
                        self.xvel += 0.25

    def collideYvel(self, yvel, room):
        """
        Check collision with static objects on y axis
        """

        if room.locked:
            solids = room.platforms + room.doors
        else:
            solids = room.platforms

        solids = solids + room.steps

        for p in solids:
            #Check collision with steps
            if isinstance(p,Step):
                #make a small collision box at bottom of player
                x,y = self.rect.midbottom
                stepCol = Rect(x, y, 1, h/4)
                stepCol.y -= h/4
                
                #Land on step
                if yvel > 0 and stepCol.colliderect(p) and not self.double_down:
                    self.rect.bottom = p.rect.top
                    self.onGround = True
                    self.yvel = 0

                    #fall damage
                    if yvel >= 15*sizeMultiplier:
                        self.life -= 0.5

            #check collisions with other platforms
            elif sprite.collide_rect(self, p):

                    #collision top
                    if yvel < 0:
                        self.rect.top = p.rect.bottom
                        self.yvel = 0
                        self.holdingJump = False

                    #collision bottom
                    if yvel > 0:
                        self.rect.bottom = p.rect.top
                        self.onGround = True
                        self.yvel = 0
                        
                        #fall damage
                        if yvel >= 15*sizeMultiplier:
                            self.life -= 0.5

    def dynamicCollide(self, room):
        """
        check collisions with game objects.
        """

        #check collision with consumables (hearts, keys, etc)
        for c in room.consumables:
            if sprite.collide_rect(self, c):

                #key
                if isinstance(c,Key):
                    self.keys += 1
                    room.shadows.remove(c.shadow)
                    room.consumables.remove(c)  

                #key
                if isinstance(c,Money):
                    self.money += 1
                    room.shadows.remove(c.shadow)
                    room.consumables.remove(c)

                #heart
                if isinstance(c,Heart):
                    lifeDiff = self.lives - self.life

                    #make sure lifepoints is <= heartcontainers
                    if lifeDiff >= 0.5:
                        if lifeDiff == 0.5:
                            self.life += 0.5
                        else:
                            self.life += 1
                        
                        room.shadows.remove(c.shadow)
                        room.consumables.remove(c)
        
        #Check collision with chests
        for u in room.unlockables:
            if sprite.collide_rect(self, u):
                #chests
                if isinstance(u,Chest) and not isinstance(u,LockedChest) and u.locked:
                    u.unlock(self.items, room)

                elif isinstance(u,LockedChest) and self.keys > 0  and u.locked:
                    self.keys -= 1
                    u.unlock(self.items, room)

        #check collision with items
        for i in room.items:
            if sprite.collide_rect(self, i):

                #apply powerup to player
                i.apply(self)
                self.holdingItem = i

        #check collision with enemies
        for e in room.enemies:
            if not e.isDead() and not self.hit:
                if sprite.collide_mask(self, e):
                    if not self.hit:
                        if self.faceright:
                            self.xvel = -5*sizeMultiplier
                        else:
                            self.xvel = 5*sizeMultiplier

                        self.life -= 0.5
                        self.hit = True

        #check collision with hazards
        for h in room.hazards:
            if sprite.collide_rect(self, h):
                if not self.hit:
                    self.life -= 0.5
                    self.hit = True

        #check collision with portal
        if room.portal:
            if sprite.collide_rect(self, room.portal):
                self.inPortal = True
        