#include <windows.h>
#include <string>
#include <stdio.h>
#include <vector>

#include <chrono>
#include <thread>

#include "glew.h"

#include <gtc/matrix_transform.hpp>
#include <vec3.hpp>

using namespace glm;

#include "shaders.h"

uint WIDTH;
uint HEIGHT;

GLuint compileVertexShader() {
	GLuint vertexShader = glCreateShader(GL_VERTEX_SHADER);

	glShaderSource(vertexShader, 1, &vertexShaderSource, NULL);
	glCompileShader(vertexShader);

	GLint success;
	glGetShaderiv(vertexShader, GL_COMPILE_STATUS, &success);

	if (!success)
	{
		GLchar infoLog[512];
		char str[512] = { '\0' };
		glGetShaderInfoLog(vertexShader, 512, NULL, infoLog);
		sprintf_s(str, "Vertex shader compilation failed: %s\n", infoLog);
		OutputDebugStringA(str);
		return 0;
	}
	else {
		return vertexShader;
	}
}

GLuint compileFragmentShader() {
	GLuint fragmentShader = glCreateShader(GL_FRAGMENT_SHADER);

	glShaderSource(fragmentShader, 1, &fragmentShaderSource, NULL);
	glCompileShader(fragmentShader);

	GLint success;
	glGetShaderiv(fragmentShader, GL_COMPILE_STATUS, &success);
	if (!success)
	{
		GLchar infoLog[512];
		char str[512] = { '\0' };
		glGetShaderInfoLog(fragmentShader, 512, NULL, infoLog);
		sprintf_s(str, "Fragment shader compilation failed: %s\n", infoLog);
		OutputDebugStringA(str);
		return 0;
	}
	else {
		return fragmentShader;
	}
}

GLuint shaderProgram;
GLuint VAO;

void linkShaders(GLuint vertexShader, GLuint fragmentShader) {
	shaderProgram = glCreateProgram();
	glAttachShader(shaderProgram, vertexShader);
	glAttachShader(shaderProgram, fragmentShader);
	glLinkProgram(shaderProgram);

	GLint success;
	glGetProgramiv(shaderProgram, GL_LINK_STATUS, &success);
	if (!success) {
		GLchar infoLog[512];
		char str[512] = { '\0' };
		glGetProgramInfoLog(fragmentShader, 512, NULL, infoLog);
		sprintf_s(str, "Shader program linking failed: %s\n", infoLog);
		OutputDebugStringA(str);
	}
	glDeleteShader(vertexShader);
	glDeleteShader(fragmentShader);
}

// Set up vertex data (and buffer(s)) and attribute pointers
vec3 v1(-0.5f, -0.5f, 0.0f);
vec3 v2(0.5f, -0.5f, 0.0f);
vec3 v3(0.0f, 0.5f, 0.0f);

// rgb(244, 95, 66)
// rgb(65, 155, 244)
// rgb(169, 244, 65)

GLfloat Colors[] = {
	(65.0f / 255.0f),(155.0f / 255.0f), (244.0f / 255.0f),
	(65.0f / 255.0f),(155.0f / 255.0f), (244.0f / 255.0f),
	(169.0f / 255.0f), (244.0f / 255.0f), (65.0f / 255.0f),
};

void SetupData() {
	std::vector<vec3> vertices;
	vertices.push_back(v1);
	vertices.push_back(v2);
	vertices.push_back(v3);

	GLuint VBO, colors;
	glGenVertexArrays(1, &VAO);
	glBindVertexArray(VAO);

	glGenBuffers(1, &VBO);
	glBindBuffer(GL_ARRAY_BUFFER, VBO);
	glBufferData(GL_ARRAY_BUFFER, sizeof(vec3) * vertices.size(), vertices.data(), GL_DYNAMIC_DRAW);
	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 3 * sizeof(GLfloat), (GLvoid*)0);
	glEnableVertexAttribArray(0);

	glGenBuffers(1, &colors);
	glBindBuffer(GL_ARRAY_BUFFER, colors);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 9, Colors, GL_DYNAMIC_DRAW);
	glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 3 * sizeof(GLfloat), (GLvoid*)0);
	glEnableVertexAttribArray(1);

	glBindBuffer(GL_ARRAY_BUFFER, 0); // Note that this is allowed, the call to glVertexAttribPointer registered VBO as the currently bound vertex buffer object so afterwards we can safely unbind
	glBindVertexArray(0); // Unbind VAO (it's always a good thing to unbind any buffer/array to prevent strange bugs)
}

static wchar_t szWindowClass[] = L"graffathon-demo";
static wchar_t szTitle[] = L"Win32 OpenGL";

HINSTANCE hInst;
HGLRC glctx;

LRESULT CALLBACK WndProc(_In_ HWND hWnd, _In_ UINT uMsg, _In_ WPARAM wParam, _In_ LPARAM lParam) {
	switch (uMsg)
	{
	case MM_MCINOTIFY:
		PostMessage(hWnd, WM_DESTROY, 0, 0);
		break;
	case WM_CHAR: {
		char str[32] = { '\0' };
		sprintf_s(str, "%u\n", wParam);
		OutputDebugStringA(str);
		if (wParam == 0x1B) { // esc
			PostMessage(hWnd, WM_DESTROY, 0, 0);
		}
		break;
	}
	case WM_CREATE: {
		PIXELFORMATDESCRIPTOR pfd =
		{
			sizeof(PIXELFORMATDESCRIPTOR),
			1,
			PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER,    // Flags
			PFD_TYPE_RGBA,        // The kind of framebuffer. RGBA or palette.
			32,                   // Colordepth of the framebuffer.
			0, 0, 0, 0, 0, 0,
			0,
			0,
			0,
			0, 0, 0, 0,
			24,                   // Number of bits for the depthbuffer
			8,                    // Number of bits for the stencilbuffer
			0,                    // Number of Aux buffers in the framebuffer.
			PFD_MAIN_PLANE,
			0,
			0, 0, 0
		};
		HDC hdc = GetDC(hWnd);

		int pixelFormat = ChoosePixelFormat(hdc, &pfd);
		SetPixelFormat(hdc, pixelFormat, &pfd);

		glctx = wglCreateContext(hdc);
		wglMakeCurrent(hdc, glctx);

		char str[64] = { '\0' };
		sprintf_s(str, "INFO: OpenGL Version: %s\n", glGetString(GL_VERSION));
		OutputDebugStringA(str);

		GLenum err = glewInit();
		if (GLEW_OK != err)
		{
			OutputDebugStringA((const char*) glewGetErrorString(err));
			PostMessage(hWnd, WM_DESTROY, 0, 0);
		}

		glViewport(0, 0, WIDTH, HEIGHT);
		GLuint vertexShader = compileVertexShader();
		GLuint fragmentShader = compileFragmentShader();

		if (vertexShader && fragmentShader) {
			linkShaders(vertexShader, fragmentShader);
		}
		else {
			PostMessage(hWnd, WM_DESTROY, 0, 0);
		}

		break;
	}
	case WM_DESTROY: {
		HDC hdc = GetDC(hWnd);
		wglMakeCurrent(hdc, NULL);
		wglDeleteContext(glctx);

		PostQuitMessage(0);
		break;
	}
	default:
		return DefWindowProc(hWnd, uMsg, wParam, lParam);
		break;
	}
	return 0;
}


void RenderFrame() {
	SetupData();

	// Render
	glClearColor(0.2f, 0.3f, 0.3f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT);

	// Draw our first triangle
	glUseProgram(shaderProgram);
	glBindVertexArray(VAO);
	glDrawArrays(GL_TRIANGLES, 0, 3);
	glBindVertexArray(0);
}

void Rotate() {
	mat4 rotationMat(1);
	rotationMat = rotate(rotationMat, 0.01f, vec3(0.0, 1.0, 0.0));
	v1 = vec3(rotationMat * vec4(v1, 1.0));
	v2 = vec3(rotationMat * vec4(v2, 1.0));
	v3 = vec3(rotationMat * vec4(v3, 1.0));
}

void Rotate2() {
	mat4 rotationMat(1);
	rotationMat = rotate(rotationMat, 0.01f, vec3(0.7071, 0.0,  0.7071));
	v1 = vec3(rotationMat * vec4(v1, 1.0));
	v2 = vec3(rotationMat * vec4(v2, 1.0));
	v3 = vec3(rotationMat * vec4(v3, 1.0));
}

void Scale() {
	mat4 scalingMat(1);
	float factor = 0.995f;
	scalingMat = scale(scalingMat, vec3(factor, factor, factor));
	v1 = vec3(scalingMat * vec4(v1, 1.0));
	v2 = vec3(scalingMat * vec4(v2, 1.0));
	v3 = vec3(scalingMat * vec4(v3, 1.0));
}

int CALLBACK WinMain(_In_ HINSTANCE hInstance, _In_ HINSTANCE hPrevInstance, _In_ LPSTR lpCmdLine, _In_ int nCmdShow) {
	WNDCLASSEX wcex;

	wcex.cbSize = sizeof(WNDCLASSEX);
	wcex.style = CS_HREDRAW | CS_VREDRAW;
	wcex.lpfnWndProc = WndProc;
	wcex.cbClsExtra = 0;
	wcex.cbWndExtra = 0;
	wcex.hInstance = hInstance;
	wcex.hIcon = LoadIcon(hInstance, IDI_APPLICATION);
	wcex.hCursor = LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
	wcex.lpszMenuName = NULL;
	wcex.lpszClassName = szWindowClass;
	wcex.hIconSm = LoadIcon(wcex.hInstance, IDI_APPLICATION);

	if (!RegisterClassEx(&wcex))
	{
		return EXIT_FAILURE;
	}

	hInst = hInstance; // Store instance handle in our global variable  

	HMONITOR hmon = MonitorFromWindow(GetDesktopWindow(), MONITOR_DEFAULTTONEAREST);
	MONITORINFO mi = { sizeof(mi) };
	if (!GetMonitorInfo(hmon, &mi)) {
		OutputDebugStringA("Could not get monitor info");
		return EXIT_FAILURE;
	}

	WIDTH = mi.rcMonitor.right - mi.rcMonitor.left;
	HEIGHT = mi.rcMonitor.bottom - mi.rcMonitor.top;

	HWND hWnd = CreateWindow(
		szWindowClass, 
		szTitle,
		WS_POPUP | WS_VISIBLE,
		mi.rcMonitor.left,
		mi.rcMonitor.top,
		WIDTH,
		HEIGHT,
		NULL,
		NULL,
		hInstance,
		NULL
	);

	if (!hWnd)
	{
		return EXIT_FAILURE;
	}

	ShowWindow(hWnd, nCmdShow);
	HDC hdc = GetDC(hWnd);

	ShowCursor(false);

	mciSendString(L"open \"zayaz-paradise-rush-edit-cc-by-nc-sa.mp3\" type mpegvideo alias mp3", NULL, 0, NULL);
	mciSendString(L"play mp3 notify", NULL, 0, hWnd);

	auto start = std::chrono::system_clock::now();
	
	bool quit = false;
	while (!quit) {
		MSG msg;
		switch (::MsgWaitForMultipleObjectsEx(0, NULL, 12, QS_ALLINPUT, 0))
		{
		case WAIT_OBJECT_0:
			while (::PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
			{
				if (msg.message == WM_QUIT) {
					quit = true;
				}
				::TranslateMessage(&msg);
				::DispatchMessage(&msg);
			}
		case WAIT_TIMEOUT:
			RenderFrame();
			::SwapBuffers(hdc);

			auto time = std::chrono::system_clock::now();
			auto duration_ms = std::chrono::duration_cast<std::chrono::milliseconds>(time - start).count();
			if (duration_ms > 10700) {
				Rotate();
			}
			if (duration_ms > 32150) {
				Rotate2();
			}
			if (duration_ms > 53500) {
				Scale();
			}
		}
		std::this_thread::sleep_for(std::chrono::milliseconds(16));
	}

	ShowCursor(true);
	return EXIT_SUCCESS;
}
