const MAX_HIGH_SCORES_PER_LEVEL = 5;

Demo.prototype.fetch = function (url, options = {}) {
  const MAX_RETRIES = 3;
  const BASE_DELAY = 500;
  
  const shouldRetry = (error, response, retryCount) => {
    if (retryCount >= MAX_RETRIES) return false;
    
    if (response && response.status >= 500 && response.status < 600) return true;
    
    if (error && (
      error.name === 'AbortError' || 
      error.name === 'TypeError' || 
      error.message.includes('timeout') ||
      error.message.includes('fetch')
    )) return true;
    
    return false;
  };
  
  const retryWithDelay = async (retryCount, errorOrStatus) => {
    const delay = BASE_DELAY * Math.pow(2, retryCount) + Math.random() * 1000;
    console.warn(`Request failed (${errorOrStatus}), retrying in ${Math.round(delay)}ms (attempt ${retryCount + 1}/${MAX_RETRIES + 1})`);
    await new Promise(resolve => setTimeout(resolve, delay));
    return attemptFetch(retryCount + 1);
  };
  
  const attemptFetch = async (retryCount = 0) => {
    const REQUEST_TIMEOUT = 1500;
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), REQUEST_TIMEOUT);
    
    try {
      const response = await fetch(url, {
        ...options,
        signal: controller.signal
      });
      
      clearTimeout(timeoutId);
      
      if (shouldRetry(null, response, retryCount)) {
        return retryWithDelay(retryCount, `Server error ${response.status}`);
      }
      
      return response;
    } catch (error) {
      clearTimeout(timeoutId);
      
      if (shouldRetry(error, null, retryCount)) {
        return retryWithDelay(retryCount, error.message);
      }
      
      throw error;
    }
  };
  
  return attemptFetch();
};

Demo.prototype.highScoresToMemory = function (highScores) {
  if (!Array.isArray(highScores)) {
    console.warn('Invalid high scores data:', highScores);
    return false;
  }

  for (let i = 0; i < highScores.length && i < MAX_HIGH_SCORES_PER_LEVEL; i++) {
    let invalid = false;
    if (!highScores[i].playerName || !highScores[i].score || !highScores[i].date) {
      console.warn('Invalid high score entry:', highScores[i]);
      invalid = true;
    }
    if (typeof highScores[i].score !== 'number') {
      console.warn('Score is not a number:', highScores[i].score);
      invalid = true;
    }
    if (isNaN(new Date(highScores[i].date).getTime())) {
      console.warn('Invalid date format:', highScores[i].date);
      invalid = true;
    }
    if (highScores[i].playerName.length > 30) {
      console.warn('Player name too long:', highScores[i].playerName);
      invalid = true;
    }

    if (invalid) {
      highScores.splice(i, 1);
      i--;
    }
  }

  if (highScores.length > MAX_HIGH_SCORES_PER_LEVEL) {
    window.latestHighScores = highScores.slice(0, MAX_HIGH_SCORES_PER_LEVEL);
  } else {
    window.latestHighScores = highScores;
  }

    const now = new Date();
    
    for(let i = 0; i<window.latestHighScores.length; i++)
    {
      const timeDiff = now - new Date(window.latestHighScores[i].date);

      if (timeDiff <= 10000)
        window.latestHighScores[i].blink = true;
      else
        window.latestHighScores[i].blink = false;
    }
    for(let i = window.latestHighScores.length; i<MAX_HIGH_SCORES_PER_LEVEL;i++)
    {
        window.latestHighScores[i] = {
            playerName: '',
            score: '',
            blink: false
        };
    }

    return true;
};

Demo.prototype.getHighScores = function (track) {
  const storageKey = `JML_AHAH_hs_${track}`;
  
  try {
    const stored = localStorage.getItem(storageKey);
    if (stored) {
      const parsedHighScores = JSON.parse(stored);
      const highScores = Array.isArray(parsedHighScores) ? parsedHighScores : [];

      this.highScoresToMemory(highScores);
      return highScores;
    }
  } catch (error) {
    console.warn('Failed to load high scores from localStorage:', error);
  }
  
  this.fetch(`https://ahahgame.jumalauta.graphics/api/sendHighScore?track=${encodeURIComponent(track)}`)
    .then(response => {
      if (response.ok) {
        return response.json();
      } else {
        console.warn('Failed to fetch high scores from server:', response.status, response.statusText);
        throw new Error(`Server responded with ${response.status}`);
      }
    })
    .then(serverHighScores => {
      if (Array.isArray(serverHighScores)) {
        this.highScoresToMemory(serverHighScores);
        setTimeout(() => {
          try {
            localStorage.setItem(storageKey, JSON.stringify(serverHighScores));
          } catch (error) {
            console.warn('Failed to cache high scores to localStorage:', error);
          }
        }, 0);
      }
    })
    .catch(error => {
      console.error('Error fetching high scores from server:', error);
    });

  return [];
};

Demo.prototype.addHighScore = function (track, playerName, score) {  
  const storageKey = `JML_AHAH_hs_${track}`;
  let highScores = this.getHighScores(track);

  const newScore = {
    playerName: playerName || 'JUMALAUTA',
    score: score || 0,
    date: new Date().toISOString()
  };
  
  //const isNewHighScore = highScores.length === 0 || score > highScores[0].score;
  highScores.push(newScore);
  highScores.sort((a, b) => b.score - a.score);

  if (highScores.length > MAX_HIGH_SCORES_PER_LEVEL) {
    highScores = highScores.slice(0, MAX_HIGH_SCORES_PER_LEVEL);
  }

  try {
    console.log(`High score added for ${track}: ${playerName} - ${score}`);

    let requestNotes = [];
    if (window.LevelNotes && Array.isArray(window.LevelNotes)) {
      for(let i = 0; i < window.LevelNotes.length; i++) {
        const note = window.LevelNotes[i];
        if (note.score == 0) {
          continue;
        }
        requestNotes.push({
          i: i,
          score: note.score,
          currentScore: note.currentScore || 0,
          currentTime: note.currentTime || 0,
        });
      }
    };
    const requestBody = {
      track: track,
      notes: requestNotes,
      gameSessionId: window.gameSessionId,
      ...newScore
    };

    try {
      localStorage.setItem(storageKey, JSON.stringify(highScores));
    } catch (error) {
      console.error('Failed to save high scores to localStorage:', error);
    }
    
    this.fetch('https://ahahgame.jumalauta.graphics/api/sendHighScore', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(requestBody)
    })
    .then(response => {
      if (response.ok) {
        return response.json();
      } else {
        console.warn('Failed to send high score to server:', response.status, response.statusText);
        throw new Error(`Server responded with ${response.status}`);
      }
    })
    .then(serverHighScores => {
      if (Array.isArray(serverHighScores)) {
        this.highScoresToMemory(serverHighScores);
        try {
          localStorage.setItem(storageKey, JSON.stringify(serverHighScores));
        } catch (error) {
          console.warn('Failed to update high scores in localStorage:', error);
        }
      }
    })
    .catch(error => {
      console.error('Error sending high score to server:', error);
    });
  } catch (error) {
    console.error('Failed to save high scores to localStorage:', error);
  }

  this.highScoresToMemory(highScores);
  return highScores;
};

Demo.prototype.getPlayerId = function () {
  if (window.ahahPlayerId) {
    return window.ahahPlayerId;
  }

  const storageKey = 'JML_AHAH_playerId';
  let playerId = localStorage.getItem(storageKey);
  if (!playerId) {
    try {
      playerId = crypto.randomUUID();
    } catch (error) {
      playerId = window.playerName + '-' + Math.random().toString(36).substring(2);
    }
  }

  window.ahahPlayerId = playerId;

  try {
    localStorage.setItem(storageKey, playerId);
  } catch (error) {
    console.error('Failed to save player ID to localStorage:', error);
  }

  return playerId;
};

Demo.prototype.startGame = function () {
  window.gameSessionId = undefined;
  const playerId = this.getPlayerId();
  const track = window.activeTrack;
  const playerName = window.playerName;

  if (!track || !playerName || !playerId) {
    console.log('missing track, playerName or playerId');
    return;
  }

  const requestBody = {
    track: track,
    playerName: playerName,
    playerId: playerId,
    date: new Date().toISOString()
  };

  this.fetch('https://ahahgame.jumalauta.graphics/api/startGame', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify(requestBody)
  })
  .then(response => {
    if (response.ok) {
      return response.json();
    } else {
      console.warn('Failed to send high score to server:', response.status, response.statusText);
      throw new Error(`Server responded with ${response.status}`);
    }
  })
  .then(response => {
    if (typeof response === 'object' && !Array.isArray(response)) {
      if (response.data?.gameSessionId) {
        window.gameSessionId = response.data.gameSessionId;
      }
    }
  })
  .catch(error => {
    console.error('Error communicating game session start:', error);
  });
};

Demo.prototype.sceneMainGame = function () {
  this.setScene('mainGame');
  this.addLevelNotes();

  this.loader.addAnimation([{
    object:{
        "name":"3d_models/airhorn.obj"
      }
    ,position:[{
      x:-10,
      y:-5,
      z:-1,
    }]

    ,scale:[{"uniform3d":1.5}]
    ,angle:[{             
      degreesX:0,      
      degreesY:90 ,    
      degreesZ:0 ,             
    }]
    ,fireLastFrame: false
    ,visible:()=>window.gameActive
    ,runFunction:(animation)=>{

        let yPos = 0;
        if (window.activeTrack === 'trainer') {
          animation.position[0].x = 0;
          animation.position[0].y = 0;
          animation.position[0].z = -1;
        } else {
          yPos = -5;
          animation.position[0].x = -10;
          animation.position[0].y = -5;
          animation.position[0].z = -1;
        }

        if(window.fire[0])
        {
          animation.scale[0].x = 3.0;        
          animation.scale[0].y = 3.0;
          animation.scale[0].z = 3.0;
          if(animation.fireLastFrame == false)
          {
            animation.angle[0].degreesX = Math.random()*40 -20;
            animation.angle[0].degreesY = 90+ Math.random()*40 -20;
            animation.angle[0].degreesZ = Math.random()*40 -20;
          }
          animation.fireLastFrame = true;

         // animation.angle[0].degreesZ = animation.angle[0].degreesZ + 5*Math.sin(getSceneTimeFromStart()*10);
          animation.angle[0].degreesY = animation.angle[0].degreesY + 1*Math.sin(getSceneTimeFromStart()*20);
 
        }
        else
        {
          animation.scale[0].x = 1.5; 
          animation.scale[0].y = 1.5;
          animation.scale[0].z = 1.5;
          if(animation.fireLastFrame == true)
          {
            animation.angle[0].degreesX = Math.random()*20 -10;
            animation.angle[0].degreesY = 90+ Math.random()*20 -10;
            animation.angle[0].degreesZ = Math.random()*20 -10;
          }
          animation.position[0].y = yPos + .25*Math.sin(getSceneTimeFromStart()*5);
          animation.fireLastFrame = false;
          animation.angle[0].degreesZ = animation.angle[0].degreesZ - .1*Math.sin(getSceneTimeFromStart()*2.5);
        }
      }
  }]);

window.musicplaying = false;
let framecount = 0;

window.player = [newPlayer(), newPlayer()];
window.currentMusic = null;

  // music
  
    this.loader.addAnimation({
      initFunction:(animation)=>{
          animation.prevPlay = false;
          animation.loopVol = 0;
          animation.sounds = {};
          const sounds =  ['rfmaudio.mp3','uhka.mp3','test.mp3', 'testShort.mp3', 'farjan.mp3', 'beethornen.mp3'];
          sounds.forEach((sound) => {
              const name = sound.replaceAll(/\.mp3/gi,'');
             // console.log("sound name " + name);
              const audioFile = new AudioFile();
              audioFile.load(sound);
              animation.sounds[name] = audioFile;
          });
          framecount = 0;
      },

      runFunction:(animation)=>{
        if(window.gameActive != true)
        {
          return;
        }
        if(window.gameInit == true)
        {
            window.timeCompensation = 9999;
        }
        if(!window.musicplaying && framecount > 20)
        {
          window.musicplaying = true;
          window.timeCompensation = getSceneTimeFromStart();

          if (window.activeTrack == "test")
            window.currentMusic = animation.sounds.test;

          if (window.activeTrack == "testShort")
            window.currentMusic = animation.sounds.testShort;

          if (window.activeTrack == "trainer")
            window.currentMusic = null;

          if(window.activeTrack == "rfm")
            window.currentMusic = animation.sounds.rfmaudio;

          if(window.activeTrack == "uhka")
            window.currentMusic = animation.sounds.uhka;

          if(window.activeTrack == "farjan")
            window.currentMusic = animation.sounds.farjan;

          if(window.activeTrack == "beethoven")
            window.currentMusic = animation.sounds.beethornen;
          
          window.currentMusic?.play();
        }

        framecount++;
      }
  });

  // sfx

  window.playerAmount = 1;
  window.maxPlayers = 2;

  for(let i = 0; i<window.maxPlayers;i++)
  {
    this.loader.addAnimation({
        initFunction:(animation)=>{
            animation.prevPlay = false;
            animation.loopVol = 0;
            animation.sounds = {};
            const sounds =  ['airhorn_start.mp3','airhorn_loop.mp3'];
            sounds.forEach((sound) => {
                const name = sound.replaceAll(/\.mp3/gi,'');
              // console.log("sound name " + name);
                const audioFile = new AudioFile();
                audioFile.load(sound);
                animation.sounds[name] = audioFile;
            });
        },

        runFunction:(animation)=>{
          if(window.gameActive != true)
          {
            if(animation.sounds.airhorn_loop.isPlaying())
              animation.sounds.airhorn_loop.stop();     
              
            window.fire[0] = false;
            return;
          }
          if(window.gameInit == true)
          {
              animation.sounds.airhorn_start.setVolume(1);
              animation.sounds.airhorn_start.setLoop(false);
              animation.sounds.airhorn_loop.setLoop(true);
              animation.sounds.airhorn_loop.setVolume(0);
              animation.sounds.airhorn_loop.stop();
              animation.sounds.airhorn_start.stop();
              animation.loopVol = 0;
              window.player[0].prevFrameHit = false;
              window.player[0].prevFrameRelease = false;
          }

          if(window.fire[0] )
          {
            if(!window.player[0].prevFrameHit)
            {
              window.player[0].currentHitActivatedNote = false;
              animation.sounds.airhorn_start.setVolume(1);
              animation.sounds.airhorn_start.play();
              animation.sounds.airhorn_start.setLoop(false);
              animation.sounds.airhorn_loop.setLoop(true);
              animation.sounds.airhorn_loop.play();

              animation.loopVol = 0;
              
              animation.sounds.airhorn_loop.setVolume(animation.loopVol);
              window.player[0].hitTime = getSceneTimeFromStart()-window.timeCompensation;          

            }
            if((getSceneTimeFromStart()-window.timeCompensation) > window.player[0].hitTime+.85)
            {
              animation.loopVol = fadeIn(animation.loopVol, .1);
              animation.sounds.airhorn_loop.setVolume(animation.loopVol);
            }
            window.player[0].prevFrameHit = true;
          }
          else
          {
            window.player[0].prevFrameHit = false;
          }

          if(!window.fire[0])
          {
            if(!window.player[0].prevFrameRelease)
            {
              animation.sounds.airhorn_start.stop();
              animation.sounds.airhorn_loop.stop();
              window.releaseTime = getSceneTimeFromStart()-window.timeCompensation;

            }        
            window.player[0].prevFrameRelease = true;
          }
          else
          {
            window.player[0].prevFrameRelease = false;

          }
        }
    });
  }

// centerline
  this.loader.addAnimation([{
    start: 0,
    image: { name: "images/noteblock.png" },
    perspective: "2d",
    position: [{ x: -0.25, y: 0, z: 0 }], 
    color: [{ r: 1.0, g: 1.0, b: 1.0 }],
    scale: [{ x: 0.1, y: 5.0 }],
    material:{depthWrite:true}
  }]);


window.perfect = .05;
window.good = .075;
window.ok = .1;
window.bad = .2;

window.currentNoteIndex = 0;
window.noteActivated = false;
window.activatedNoteTime = 0;
window.noteDeactivated = false;
window.activatedNoteDuration = 0;
window.latestNoteR = 0;
window.latestNoteG = 0;
window.latestNoteB = 0;

this.loader.addAnimation({
    runFunction:()=>{
      let currentTime = 0;

      if(window.gameActive == false)
      {
        return;
      }
      else if (window.gameInit == true)
      {
     
        window.latestNoteR = 0;
        window.latestNoteG = 0;
        window.latestNoteB = 0;
        window.noteActivated = false;
       // console.log("init");
        window.menuSelect = false;

        switch (window.activeTrack) {
          case "test":
            window.LevelNotes = window.testNotes;
            break;
          case "testShort":
            window.LevelNotes = window.testShortNotes;
            break;
          case "trainer":
            window.LevelNotes = window.trainerNotes;
            break;
          case "rfm":
            window.LevelNotes = window.rfmNotes;
            break;
          case "uhka":
            window.LevelNotes = window.uhkaNotes;
            break;
          case "farjan":
            window.LevelNotes = window.farjanNotes;
            break;
          case "beethoven":
            window.LevelNotes = window.beethovenNotes;
            break;
        }

        window.currentNoteIndex = 0;
        for (let j = 0; j < window.LevelNotes.length; j++)
        {

        window.LevelNotes[j].played = false;
        window.LevelNotes[j].score = 0;
        window.LevelNotes[j].r = 1;
        window.LevelNotes[j].g = 1;
        window.LevelNotes[j].b = 1;
        window.LevelNotes[j].active = false;
        window.LevelNotes[j].currentScore = undefined;
        window.LevelNotes[j].currentTime = undefined;
        }

        window.score = 0;
        window.tempScore = 0;

       
      }

      currentTime = getSceneTimeFromStart() - window.timeCompensation;
      // note handling
      let currentNote = window.LevelNotes[window.currentNoteIndex];

      if(currentNote.active && currentTime >= (currentNote.time))
      {
        window.tempScore = currentNote.score*(currentTime-currentNote.time);
      }

      if(currentTime >= Math.max(currentNote.time + currentNote.duration,currentNote.time + window.bad))
      {
        if(currentNote.played != true)
        {
          currentNote.score = 0;
          currentNote.played = true;
          currentNote.r = 1.0;
          currentNote.g = 0;
          currentNote.b = 0;
        }

        if(currentNote.active)
        {
          window.noteDeactivated = true;
          currentNote.active = false;
          currentNote.currentScore = currentNote.duration*20;
          currentNote.currentTime = currentTime;
          window.tempScore = 0;
          window.score += currentNote.duration*20;
        }
        if(window.currentNoteIndex < window.LevelNotes.length-1)
           window.currentNoteIndex++;

      }

      if(currentTime >= (currentNote.time + window.bad))
      {
        if(currentNote.played != true)
        {
          currentNote.score = 0;
          currentNote.played = true;
          currentNote.r = 1;
          currentNote.g = 0.25;
          currentNote.b = 0.25;
        }
      }      
      
      
      if(currentNote.played != true && window.player[0].currentHitActivatedNote == false)
      {
        if(Math.abs(currentNote.time - window.player[0].hitTime) < window.perfect)
        {
          window.latestNoteR=0;
          window.latestNoteG=1;
          currentNote.played = true;
          currentNote.score = 10;
        }
        else if(Math.abs(currentNote.time - window.player[0].hitTime) < window.good)
        {
          window.latestNoteR=1;
          window.latestNoteG=1;
          currentNote.played = true;
          currentNote.score = 8;
        }
        else if(Math.abs(currentNote.time - window.player[0].hitTime) < window.ok)
        {
          window.latestNoteR=1.0;
          window.latestNoteG=.5;
          currentNote.played = true;
          currentNote.score = 5;
        }
        else if((Math.abs(currentNote.time - window.player[0].hitTime) < window.bad))
        {
          window.latestNoteR=1;
          window.latestNoteG=.25;
          currentNote.played = true;
          currentNote.score = 2;
        }

        if(currentNote.played == true)
        {
          currentNote.active = true;
          window.latestNoteB = 0;
          window.score += currentNote.score;
          window.noteActivated = true;
          window.activatedNoteTime = currentNote.time;
          window.activatedNoteDuration = currentNote.duration;
          window.player[0].currentHitActivatedNote = true;


        }
      }

      if(currentNote.active && window.releaseTime > window.player[0].hitTime)
      {
        currentNote.active = false;
        currentNote.currentScore = window.tempScore;
        currentNote.currentTime = currentTime;
        window.score += window.tempScore;
        window.tempScore = 0;
      }

      /* if(window.player[0].hitTime < window.releaseTime)
      {
        window.player[0].hitTime = 0.0;
      }
        */
    }
  });



  this.loader.addAnimation([{
    start: 0,
    runFunction: (animation) => {
      window.notesOnDisplay = {};
    }
  }]);

  for (let i = 0; i < 30; i++) {
  this.loader.addAnimation([{
    start: 0,
    image: { name: "images/noteblock.png" },
    perspective: "2d",
    position: [{ x: 100, y: 0, z: 0 }], // will be overwritten
    color: [{ r: 1.0, g: 1.0, b: 1.0 }],
    scale: [{ x: 0, y: 1.0 }],
    material:{depthWrite:true},
    noteIndex: i,
    noteIndexStart: 0,
    runFunction: (animation) => {
      if(window.gameActive == false || window.gameInit) {
        animation.ref.mesh.visible = false;
        animation.position[0].x = -100;
        animation.scale[0].x = 0;
        animation.noteIndexStart = 0;
        return;
      }

      animation.ref.mesh.visible = false;

      const currentTime = getSceneTimeFromStart() - window.timeCompensation;
      for(let noteIndex = animation.noteIndexStart; noteIndex < window.LevelNotes.length; noteIndex++) {
        if (window.notesOnDisplay[noteIndex]) {
          continue;
        }
        window.notesOnDisplay[noteIndex] = true;
        const note = window.LevelNotes[noteIndex];  

        const timeUntilHit = note.time - currentTime;
        if (timeUntilHit > 5) {
          animation.ref.mesh.visible = false;
          continue;
        }
        const scrollSpeed = 900; // px/sec
        const positionUnitPx = 1920; // 1.0 in position units = 1920px
        const pixelOffset = timeUntilHit * scrollSpeed;
        const noteWidthPx = note.duration * scrollSpeed;
        const pixelOffsetPos = pixelOffset / positionUnitPx;
        const noteWidthPos = noteWidthPx / positionUnitPx;
        const centerX = -.25 + pixelOffsetPos + noteWidthPos / 2;
        const scaleX = noteWidthPx / 100;

        if (centerX-scaleX/2.0 > 0.5 || centerX+scaleX/2.0 < -0.5) {
          animation.ref.mesh.visible = false;
          animation.position[0].x = -100;
          animation.scale[0].x = 0;
          continue;
        }
        animation.ref.mesh.visible = true;
        animation.noteIndexStart = noteIndex;

        animation.position[0].x = centerX;
        animation.scale[0].x = scaleX;
        animation.color[0].r = window.LevelNotes[noteIndex].r;
        animation.color[0].g = window.LevelNotes[noteIndex].g;
        animation.color[0].b = window.LevelNotes[noteIndex].b;
        break;
      }
           
      }
    }]);
  }

  window.activeHitIndex = 0;
  window.prevActiveHitIndex = 0;
  for (let i = 0; i< 8; i++)
  {
    this.loader.addAnimation([{
      start: 0,
      image: { name: "images/noteblock.png" },
      perspective: "2d",
      position: [{ x: -100, y: 0, z: 0 }], // will be overwritten
      color: [{ r: .0, g: .0, b: 1.0 }],
      scale: [{ x: 0, y: 1.0 }],
      material:{depthWrite:true},
      index: i,
      hitTime: 0,
      releaseTime: 0,
      activated: false,
      runFunction: (animation) => {
        if(window.gameActive == false)
          return;
        let deActivate = false;
        
        if(window.gameInit)
        {
          animation.position[0].x = .100;
           animation.scale[0].x = 0;
          animation.color[0].r = 1;
          animation.color[0].g = 1;
          animation.color[0].b = 1;
          animation.hitTime = 0.0;
          animation.activated = false;
          animation.releaseTime = 0.0;
        //  console.log("init");
          return;
        }

        const currentTime = getSceneTimeFromStart() - window.timeCompensation;

        if(animation.activated && window.player[0].hitTime < window.releaseTime  )
        {
          if(noteDeactivated)
          {
            animation.releaseTime = window.activatedNoteDuration;
          }
          animation.activated = false;
          // if hit indices are same this is release
          if(window.activeHitIndex == window.prevActiveHitIndex)
          {
            window.activeHitIndex++;
            if(window.activeHitIndex > 7)
            {
              window.activeHitIndex = 0;
            }
          }
        }
        else if(window.activeHitIndex == animation.index)
        {
          // only recalculate for active hit index
          
          if(window.noteActivated)
          {
            animation.color[0].r = window.latestNoteR;
            animation.color[0].g = window.latestNoteG;
            animation.color[0].b = window.latestNoteB;
            
            animation.hitTime = window.activatedNoteTime;
            window.prevActiveHitIndex = animation.index;
            window.noteActivated = false;
            animation.activated = true; 
          }
          // update release time for active
          if(animation.activated)
          {
            animation.releaseTime = currentTime - animation.hitTime;
            if(animation.releaseTime <= window.perfect)
              animation.releaseTime = window.perfect;

            if(animation.releaseTime > window.activatedNoteDuration)
              animation.releaseTime = window.activatedNoteDuration;
          }
        }
        

       // console.log("hitTime:" + animation.hitTime + "release time" + animation.releaseTime);
        const scrollSpeed = 900; // px/sec
        const positionUnitPx = 1920; // 1.0 in position units = 1920px
        const timeUntilHit = animation.hitTime - currentTime;
        const pixelOffset = timeUntilHit * scrollSpeed;
        const noteWidthPx = animation.releaseTime * scrollSpeed;
        const pixelOffsetPos = pixelOffset / positionUnitPx;
        const noteWidthPos = noteWidthPx / positionUnitPx;
        const centerX = -.25 + pixelOffsetPos + noteWidthPos / 2;
        const scaleX = noteWidthPx / 100;

        animation.position[0].x = centerX;
        animation.scale[0].x = scaleX;
        //animation.color[0].r = window.LevelNotes[animation.noteIndex].r;
        //animation.color[0].g = window.LevelNotes[animation.noteIndex].g;
        //animation.color[0].b = window.LevelNotes[animation.noteIndex].b;
            
      }
    }]);
  }

  const demo = this;

  this.loader.addAnimation({
      runFunction:()=>{
        
        if(window.gameActive == false || window.levelEnd == true)
          return;

        if(window.gameInit)
          window.gameInit = false;

        let levelduration = window.LevelNotes[window.LevelNotes.length-1].time + window.LevelNotes[window.LevelNotes.length-1].duration + 1.0;
        let currentTime = getSceneTimeFromStart()-window.timeCompensation;
        window.noteDeactivated = false;
        window.noteActivated = false;

        if(currentTime > levelduration)
        {

          window.finalScore =  Math.floor(window.score);

          demo.addHighScore(window.activeTrack, window.playerName, window.finalScore);

          window.levelEndTime = 9999;
          window.levelEnd = true;
          window.endxplosionplayed = false;
          window.gameActive = false;
          window.gameInit = false;
          window.musicplaying = false;
          window.currentMusic?.stop();    
          demo.createHighScoreDiv(); 
          
        }
      }
  });

  this.addScoreCalculator();

  function fadeIn(volume, time)
  {
    volume += getDeltaTime() / time;
    if (volume > 1.0)
      return 1.0;
    else
      return volume;
  }

  function newPlayer() {
  return {
    hitTime: 0,
    currentHitActivatedNote: false,
    prevFrameHit: false,
    prevframeRelease: false,
  };
}

};

