; msweep - small minesweeper - alrj - 2024
; Inspired by Chris Mumma's 253 bytes minesweeper game

; Redistribution and use in source and binary forms, with or without
; modification, are permitted without limitation.

BITS 16

; Color theme.
; Just uncomment the one you like the most :-)
COLOR_THEME     equ     -8              ; This makes an orange to yellow theme with good contrast for flags and mines
; COLOR_THEME     equ     -41             ; This makes digits fall into the 8-15. Colorfull, but less contrasted mines.

; Difficulty
; Number of mines to put in the game.
; 10 is only good for debugging. Super easy, very fast game
; 25 is easy
; 35 is medium
; 50 is difficult, usually with many places where guessing is necessary
MAP_MINES       equ     35              ; Number of mines


MAP_PHYSW       equ     20              ; Physical width of mine map in memory
MAP_WIDTH       equ     19              ; Last column of the map is border
MAP_HEIGHT      equ     12              ; Number of rows
FREE_CELLS      equ     MAP_HEIGHT * MAP_WIDTH - MAP_MINES


; Build as a .COM file
org 100h

section .text

start:
; Setup screen mode 13h
        mov al, 13h                     ; Setup mode 13h because
        int 10h                         ; I want an arrow mouse cursor!

        int 1Ah                         ; Get system time into CX:DX for pseudorandom number generator
                                        ; Only DX will be used.

%assign SetupScreenSize ($-start)
%warning Setup Screen: SetupScreenSize bytes

; The low nibble of a cell indicates how many mines are present in its 8 neighbours.
; The high nibble is used as a bit field describing the state of the cell.
; Bit 7: set if the cell is part of the map border, that is outside the map.
; Bit 6: set if the cell has been revealed
; Bit 5: set if the cell contains a bomb
; Bit 4: set if the cell has been flagged by the player

SetupMap:
        ; Assert CS = DS = ES
        ; "Blind" map prepare, fill a "large" memory chunk as if it was entirely outside of the map
        mov di, premap
        mov al, 80h                     ; bit 7 set = outside of the map
        mov ch, 2                       ; Whatever size, between 512 and 767, as long as it is too big
        rep stosb                       ; Pre-fill with 80h's, clear CX

        ; Build map with empty covered cells
        mov di, mapData                 ; This is the address of the first actual cell of the map
        push di                         ; And save for later
        xor ax, ax                      ; Clear AX, not just AL, in preparation for int 33h just after this loop

        mov bl, MAP_HEIGHT              ; (BX is zero at program start)
.setupline:
        mov cl, MAP_WIDTH
        rep stosb                       ; Fill a line with empty cells
        inc di                          ; Leave a border cell and advance to the start of next line
        dec bx
        jnz .setupline

        int 33h                         ; Init mouse while AX is zero, now that AX, BX, CX can be trashed

%assign PrepareMapSize ($-SetupMap)
%warning Prepare Map (and mouse): PrepareMapSize bytes

                                        ; now BH, CX should be clear
SetupMines:
        ; Place the mines in the map
        pop di                          ; DI -> mapData
        mov cl, MAP_MINES
        xchg ax, dx                     ; PRNG seed into AX
.setupmine:
        rol ax, 1                       ; \ Very simple
        xor ax, cx                      ; / PRNG
        mov bl, al                      ; BH is already zero

        test byte [di+bx], 0A0h         ; Is any of bit 7 or 5 set?
        jnz .setupmine                  ; then it's a border or a mine, try another spot

        or byte [di+bx], 0x20           ; set bit 5 to mark a mine

        mov dx, incCell
        call NeighApply                 ; Increment mine count in the 8 cells around

        loop .setupmine
        ; This loop needs AX, BH (0), CX, DI preserved.
%assign SetupMinesSize ($-SetupMines)
%warning Setup Mines: SetupMinesSize bytes


Init:
        call DrawMap                    ; DrawMap, which ends with ShowMouse

%assign InitSize ($-Init)
%warning Init size: InitSize bytes

GameLoop:
        ; Uncomment these to support exitting by pressing the escape key
        ; This takes 5 bytes:
        ;in al, 60h
        ;dec ax
        ;jz Exit                        ; Exit on ESC key pressed

        mov bx, 1                       ; Right button
        call ReadMouse                  ; Returns with CH=0
        js .notright
        xor byte [di+bx], 10h           ; On right click, flip the flagged bit
        jmp .redraw
.notright:

        xor bx, bx                      ; Left button
        call ReadMouse                  ; Returns with CH=0
        js GameLoop

        call FloodReveal                ; Reveal the cell, and the ones around if its value is zero.
        ; FloodReveal preserves everything except AL

.redraw:
        ; We get to .redraw if a mouse button was pressed, with AX containing the number
        ; of times the button was pressed. It is very likely less than 256, so
        ; AH is already 0.
        pusha                           ; BX must be preserved
        mov al, 2
        int 33h                         ; Mouse function 2 : Hide mouse cursor
        call DrawMap                    ; DrawMap ends with ShowMouse
        popa

        ; Test for bomb
        test byte [di+bx], 60h          ; Test for Uncovered and Bomb flags
        jz .none                        ; If none is set, no reason to lose
        jpe Lost                        ; Parity even if both are set: it is a bomb and it is uncovered

.none:
        cmp [uncovered], ch             ; Check the count of uncovered empty cells (against CH which is 0)
        jnz GameLoop                    ; If it is down to zero, player won.

Exit:
Lost:
        ; The only way to distinguish a win from a loss, is whether the last cell clicked
        ; is now showing a bomb or not.
        ; Hopefully, the color of the bombs is contrasted enough.

        ; Uncomment the next two lines to wait for a keypress when game is finished.
        ; This takes 4 bytes:
        ;xor ax, ax
        ;int 16h                        ; Wait for keypress (or eat the ESC if it was pressed)

        ; Uncomment the next two lines to restore video mode 03h on exit.
        ; It is better to uncomment the previous lines as well in this case, otherwise
        ; it becomes impossible to see if the player won or not, because it blanks the screen.
        ; This takes 5 bytes:
        ;mov ax, 3
        ;int 10h                        ; Restore video mode

        ; If next 'ret' is commented out, fall through incCell to exit
        ;ret                            ; return to DOS

%assign GameLoopSize ($-GameLoop)
%warning Main Game loop: GameLoopSize bytes


;======================================================================
; incCell - Increase the bomb count by one in the cell at [DI+BX]
; This function is intended to be called via NeighApply, during the setup of
; the map.
incCell:
        inc byte [di+bx]
        ret
%assign IncCellSize ($-incCell)
%warning Inc Cell: IncCellSize bytes


;======================================================================
; DrawMap - Refresh the map on screen
; Input invariants:
;       - (AX: number of mouse button presses. AH is likely zero!)
;       - BX offset to cell of last click (left or right)
;       - (CX: column >> 5: maximum 20)
;       - (DX: row >> 4: maximum 12)
;       - DI points to mapData
; Output invariants:
;       - AH = 0,  AL = 1
;       - BH = 0,  BL = color used for last cell
;       - CH = 0,  CL = 1
;       - DX undefined
;       - DI still points to mapData
;       - SI points to bottom border row of map

DrawMap:
        mov dh, 01h                     ; setup cursor on second row (1)

        mov si, di                      ; Have SI point to the map
.lines:
        mov dl, 01h                     ; setup cursor on second column (1)

DrawCell:

        ; Move cursor to row DH, column DL
        mov ah, 2                       ; fn 02h: Move cursor
        xor bx, bx                      ; BH must be zero in graphics mode
        int 10h

        lodsb                           ; Load cell content

.GetAL:
        mov ah, 0fh
        and ah, al                      ; Get number of neighbour bombs in ah
        jz .empty
        add ah, 30h                     ; As the ASCII code of the digit, if not zero
.empty:
        test al, 0a0h                   ; Is it a bomb or a border ?
        js .nextline                    ; If a border, this line is done
        jz .notbomb
        mov ah, '*'
.notbomb:
        test al, 040h                   ; Is it uncovered ?
        jnz .uncovered
        mov ah, 0DBh                    ; Square
.uncovered:
        test al, 010h                   ; Is it a flag ?
        jz .notflag
        mov ah, 13 ;'P'
.notflag:

%assign GetALSize ($-.GetAL)

.draw:
        mov al, ah                      ; Get character back into AL
        mov bl, COLOR_THEME             ; Offset into color palette
        add bl, al
        ; fg/bg colors in BX. Background color (BH) is always 0 (black).
        mov cl, 1                       ; Only one character to print (CH expected to be zero)
        mov ah, 09h
        int 10h

%assign DrawCellSize ($-DrawCell)

        inc dx                          ; Advance by two columns on screen
        inc dx                          ; Same as "add dl, 2" but one byte shorter
        jmp DrawCell                    ; Draw next cell

.nextline:
        add dh, 2                       ; Two rows down
        cmp dh, (MAP_HEIGHT * 2 + 1)
        jb DrawMap.lines

        ; Fall through to ShowMouse to return

%assign DrawMapSize ($-DrawMap)
%warning Draw Map: DrawMapSize bytes (DrawCellSize bytes for cell drawing, GetALSize bytes to get character)


; ========
; Mouse handling

ShowMouse:
        mov ax, 1                       ; Mouse function 1: Show mouse cursor
        jmp MouseInt                    ; MouseInt trashes AX, BX, CX, DX but it's fine

ReadMouse:                              ; Expects BX being set to 0 or 1 for left or right button
        mov ax, 5                       ; Mouse function 5: Return button press data

MouseInt:
        int 33h

        ; Remove half a character width and height to make the cursor
        ; hot spot more centered on a cell.
        ; This takes 6 bytes:
        sub cx, 8
        sub dx, 4

        shr cx, 5                       ; convert column on screen to column in map ( / 32)
        shr dx, 4                       ; convert line on screen to row in map ( / 16)
        imul ax, dx, 20                 ;

        add ax, cx                      ; offset of last click spot in mapData
        dec bx                          ; Set Sign Flag if no button has been pressed
        xchg ax, bx                     ; button presses (-1) in AX, offset in BX
        ret

%assign MouseSize ($-ShowMouse)
%warning Mouse: MouseSize bytes


;======================================================================
; FloodReveal - Reveal the area bordering cells with no bomb neighbours
; Reveal the cell at [DI+BX] (if eligible), then if this cell has zero neighbours
; mines, recurse to reveal its neighbours.
; This function is first called upon left click on a cell, but is then
; recursively called via NeighApply (by falling through to it)
FloodReveal:
        mov al, [di+bx]                 ; Load cell content
        test al, 0D0h                   ; Check if border or revealed or flagged
        jnz return                      ; If any of these, then do nothing
        or byte [di+bx], 40h            ; otherwise, reveal the cell
        dec byte [uncovered]            ; and account for it

        test al, 2Fh                    ; Check flag bit and neighbours count
        jnz return                      ; If not zero, no flooding from here

        ; We revealed a zero, recurse to neighbours
        mov dx, FloodReveal             ; Function to call

%assign FloodRevealSize ($-FloodReveal)
%warning Flood reveal: FloodRevealSize bytes


;======================================================================
; NeighApply - Apply a function to all 8 neighbours of a cell
; Call the function at [DX] for each neighbour of the cell pointed by [DI+BX]
; with BX as the offset from DI to the neighbour
; Expects CH to be 0
NeighApply:
        pusha                           ; Needs to preserve DI, SI and CX across recursive calls
        add di, bx                      ; rebase DI as our new starting point

        mov si, nb_offsets              ; List of offsets to the 8 neighbours (in data, below)
        mov cl, 8                       ; Length of the list
.nbloop:
        lodsb                           ; Load offset to neighbour
        cbw                             ; As word
        xchg ax, bx                     ; Need it in BX for addressing
        call dx                         ; Indirect call to function to apply
        loop .nbloop
popaAndRet:
        popa
return:
        ret
%assign NeighApplySize ($-NeighApply)
%warning Neighbour Apply: NeighApplySize bytes



%assign TotalCodeSize ($-start)
%warning Total Code size: TotalCodeSize

;section .data
data:
vanity          db      'alrj'
nb_offsets      db      -21, -20, -19, -1, 1, 19, 20, 21
uncovered       db      FREE_CELLS


%assign DataSize ($-data)
%warning Total Data Size: DataSize bytes

section .bss
premap          resb MAP_PHYSW*2        ; Top border
mapData         resb (MAP_PHYSW * MAP_HEIGHT)
postmap         resb (768 - ($-premap)) ; Enough space for "blind" map prepare
