/* meshSlicer.js
 * --------------
 * Script démontrant comment "casser" (slicer) un maillage en deux
 * via un plan de coupe défini par un point et une normale.
 *
 * ! DISCLAIMER !
 * - Code non testé tel quel, c'est un exemple avancé.
 * - Ne gère pas la re-calcul des normales ni des UVs correctement.
 * - Ne ferme pas la coupe (pas de bouchon).
 * - Pour un usage réel, il faut peaufiner et tester en profondeur.
 */

var MeshSlicer = pc.createScript('meshSlicer');

/** Attributs dans l'éditeur */
MeshSlicer.attributes.add('targetEntity', {
    type: 'entity',
    title: 'Target Entity',
    description: 'Entité dont on veut slicer le maillage'
});

MeshSlicer.attributes.add('planePoint', {
    type: 'vec3',
    default: [0, 0, 0],
    title: 'Plane Point',
    description: 'Un point appartenant au plan de coupe'
});

MeshSlicer.attributes.add('planeNormal', {
    type: 'vec3',
    default: [0, 1, 0],
    title: 'Plane Normal',
    description: 'La normale du plan de coupe'
});

/**
 * initialize() est appelée une fois au démarrage.
 */
MeshSlicer.prototype.initialize = function() {
    // Normaliser la normale du plan (important pour les calculs)
    this._planeNormal = new pc.Vec3(this.planeNormal.x, this.planeNormal.y, this.planeNormal.z).normalize();
    this._planePoint = new pc.Vec3(this.planePoint.x, this.planePoint.y, this.planePoint.z);

    // On s'assure qu'on ait bien un ModelComponent
    if (!this.targetEntity || !this.targetEntity.model) {
        console.warn("MeshSlicer: targetEntity non valide ou sans ModelComponent");
        return;
    }

    // Pour simplifier, on va prendre uniquement le premier meshInstance
    var meshInstances = this.targetEntity.model.meshInstances;
    if (!meshInstances || meshInstances.length === 0) {
        console.warn("MeshSlicer: Aucun meshInstance trouvé sur targetEntity");
        return;
    }
    
    // On récupère le mesh
    var mesh = meshInstances[0].mesh;
    if (!mesh) {
        console.warn("MeshSlicer: mesh non trouvé");
        return;
    }

    // Extraire les données du mesh (positions, indices, etc.)
    var meshData = this._extractMeshData(mesh);

    // Effectuer la coupe
    var sliceResult = this._sliceMesh(meshData, this._planePoint, this._planeNormal);

    // Construire deux nouveaux pc.Mesh
    var meshPositive = this._createPcMesh(sliceResult.posSide, this.app.graphicsDevice);
    var meshNegative = this._createPcMesh(sliceResult.negSide, this.app.graphicsDevice);

    // Créer deux entités pour ces deux nouveaux meshes
    var parent = this.targetEntity.parent;

    // Entité + (au-dessus ou côté "positif" du plan)
    var entityPos = new pc.Entity("Sliced_Positive");
    entityPos.addComponent("model", {
        type: "asset",
        castShadows: this.targetEntity.model.castShadows,
        receiveShadows: this.targetEntity.model.receiveShadows
    });
    parent.addChild(entityPos);

    // Entité - (en-dessous ou côté "négatif" du plan)
    var entityNeg = new pc.Entity("Sliced_Negative");
    entityNeg.addComponent("model", {
        type: "asset",
        castShadows: this.targetEntity.model.castShadows,
        receiveShadows: this.targetEntity.model.receiveShadows
    });
    parent.addChild(entityNeg);

    // Associer nos nouveaux meshes
    if (meshPositive) {
        entityPos.model.model.meshInstances[0].mesh = meshPositive;
        // Vous pouvez attribuer un material existant, par ex. :
        // entityPos.model.meshInstances[0].material = ...
    }

    if (meshNegative) {
        entityNeg.model.model.meshInstances[0].mesh = meshNegative;
        // Idem, attribuer un material au besoin
    }

    // Désactiver l'entité d'origine
    this.targetEntity.enabled = false;

    console.log("MeshSlicer: Slice terminé !");
};


/**
 * _extractMeshData(mesh)
 * Récupère les positions et indices du mesh.
 * (Ne gère pas dans cet exemple les normales ni les UVs)
 */
MeshSlicer.prototype._extractMeshData = function(mesh) {
    var vertexBuffer = mesh.vertexBuffer;
    var indexBuffer = mesh.indexBuffer[0];
    var format = vertexBuffer.getFormat();

    // Chercher la sémantique POSITION
    var positionSem = null;
    for (var i = 0; i < format.elements.length; i++) {
        var element = format.elements[i];
        if (element.name === pc.SEMANTIC_POSITION) {
            positionSem = element;
            break;
        }
    }
    if (!positionSem) {
        console.error("MeshSlicer: Impossible de trouver SEMANTIC_POSITION");
        return null;
    }

    // Extraire les positions
    var iterator = new pc.VertexIterator(vertexBuffer);
    var positions = new Array(vertexBuffer.numVertices);
    
    for (var v = 0; v < vertexBuffer.numVertices; v++) {
        var x = iterator.element[pc.SEMANTIC_POSITION].x;
        var y = iterator.element[pc.SEMANTIC_POSITION].y;
        var z = iterator.element[pc.SEMANTIC_POSITION].z;
        
        positions[v] = new pc.Vec3(x, y, z);
        
        iterator.next();
    }
    iterator.end();

    // Extraire les indices
    var indices = new Uint16Array(indexBuffer.lock());

    return {
        positions: positions,
        indices: indices
    };
};


/**
 * _sliceMesh(meshData, planePt, planeNormal)
 * Découpe la géométrie en deux séries de triangles
 * (côté positif et côté négatif du plan).
 */
MeshSlicer.prototype._sliceMesh = function(meshData, planePt, planeNormal) {
    var posSide = {
        positions: [],
        indices: []
    };
    var negSide = {
        positions: [],
        indices: []
    };

    // Tableau pour construire les positions cumulées
    // (on va recréer des listes indépendantes pour posSide et negSide)
    var positions = meshData.positions;
    var indices = meshData.indices;

    // Petit utilitaire local pour distance signée
    var signedDist = function(p) {
        // distance = dot((p - planePt), planeNormal)
        var v = p.clone().sub(planePt);
        return v.dot(planeNormal);
    };

    // Pour stocker la nouvelle liste de positions (on doit "dupliquer" quand intersection)
    // On va le faire "à l’ancienne", en reconstruisant triangle par triangle.
    // Note : on ne gère pas la fermeture (le bouchon).
    //       On ne gère pas non plus les normales/UV.
    var processTriangle = (i0, i1, i2) => {
        var p0 = positions[i0];
        var p1 = positions[i1];
        var p2 = positions[i2];

        var d0 = signedDist(p0);
        var d1 = signedDist(p1);
        var d2 = signedDist(p2);

        // On détermine de quel côté se trouve chaque vertex
        var side0 = d0 >= 0 ? 1 : -1;
        var side1 = d1 >= 0 ? 1 : -1;
        var side2 = d2 >= 0 ? 1 : -1;

        if (side0 === side1 && side1 === side2) {
            // Tout le triangle est d’un même côté
            if (side0 > 0) {
                // Copie dans posSide
                var baseIndex = posSide.positions.length;
                posSide.positions.push(p0.clone(), p1.clone(), p2.clone());
                posSide.indices.push(baseIndex, baseIndex+1, baseIndex+2);
            } else {
                // Copie dans negSide
                var baseIndex = negSide.positions.length;
                negSide.positions.push(p0.clone(), p1.clone(), p2.clone());
                negSide.indices.push(baseIndex, baseIndex+1, baseIndex+2);
            }
        } else {
            // Le triangle est coupé par le plan
            // => il faut calculer les points d’intersection
            // On va (pour simplifier) gérer tous les cas
            // en trouvant les 2 edges qui traversent le plan.
            // Il y a beaucoup de variantes… on fait basique.

            // Collecter dans un tableau
            var allP = [p0, p1, p2];
            var allD = [d0, d1, d2];
            var side = [side0, side1, side2];

            var posVerts = [];
            var negVerts = [];

            // Parcourir les 3 edges (0->1, 1->2, 2->0)
            for (var e = 0; e < 3; e++) {
                var e0 = e;
                var e1 = (e+1) % 3;

                var pA = allP[e0];
                var pB = allP[e1];
                var dA = allD[e0];
                var dB = allD[e1];
                var sA = side[e0];
                var sB = side[e1];

                if (sA >= 0) posVerts.push(pA);
                else negVerts.push(pA);

                // Intersection ?
                if (sA * sB < 0) {
                    // Edge traverse le plan
                    var t = Math.abs(dA) / (Math.abs(dA) + Math.abs(dB));
                    // On interpole la position
                    var intersect = pA.clone().lerp(pB, t);
                    // On pousse ce point d’intersection des deux côtés
                    posVerts.push(intersect);
                    negVerts.push(intersect.clone());
                }
            }

            // Maintenant, posVerts et negVerts contiennent 2 ou 3 points
            // On va trianguler ces points pour remplir posSide / negSide.

            if (posVerts.length >= 3) {
                var baseIndex = posSide.positions.length;
                posSide.positions.push(posVerts[0], posVerts[1], posVerts[2]);
                posSide.indices.push(baseIndex, baseIndex+1, baseIndex+2);
                // S’il y a 4 points (rare), on ferait un 2ᵉ triangle
                if (posVerts[3]) {
                    // Tri #2
                    posSide.positions.push(posVerts[0], posVerts[2], posVerts[3]);
                    posSide.indices.push(baseIndex+3, baseIndex+4, baseIndex+5);
                }
            }
            if (negVerts.length >= 3) {
                var baseIndex = negSide.positions.length;
                negSide.positions.push(negVerts[0], negVerts[1], negVerts[2]);
                negSide.indices.push(baseIndex, baseIndex+1, baseIndex+2);
                if (negVerts[3]) {
                    negSide.positions.push(negVerts[0], negVerts[2], negVerts[3]);
                    negSide.indices.push(baseIndex+3, baseIndex+4, baseIndex+5);
                }
            }
        }
    };

    // Parcourir tous les triangles
    for (var t = 0; t < indices.length; t += 3) {
        var i0 = indices[t];
        var i1 = indices[t+1];
        var i2 = indices[t+2];

        processTriangle(i0, i1, i2);
    }

    return {
        posSide: posSide,
        negSide: negSide
    };
};


/**
 * _createPcMesh(meshData, device)
 * Construit un pc.Mesh à partir d'un tableau de positions + indices.
 */
MeshSlicer.prototype._createPcMesh = function(meshData, device) {
    if (!meshData.positions || meshData.positions.length === 0) {
        return null; // Pas de géométrie
    }

    // Création d’un VertexFormat
    var vertexFormat = new pc.VertexFormat(device, [
        { semantic: pc.SEMANTIC_POSITION, components: 3, type: pc.TYPE_FLOAT32 }
        // On pourrait ajouter d’autres sémantiques (normales, UV, etc.) si on les gérait
    ]);

    var numVerts = meshData.positions.length;
    var vertexBuffer = new pc.VertexBuffer(device, vertexFormat, numVerts);

    // Remplir le buffer
    var iterator = new pc.VertexIterator(vertexBuffer);
    for (var i = 0; i < numVerts; i++) {
        var pos = meshData.positions[i];
        iterator.element[pc.SEMANTIC_POSITION].set(pos.x, pos.y, pos.z);
        iterator.next();
    }
    iterator.end();

    // Créer l’index buffer
    // On suppose que le nombre de triangles n’excède pas 65535 indices
    var indices = new Uint16Array(meshData.indices);
    var indexBuffer = new pc.IndexBuffer(device, pc.INDEXFORMAT_UINT16, indices.length);
    var dst = new Uint16Array(indexBuffer.lock());
    dst.set(indices);
    indexBuffer.unlock();

    // Enfin, construire le Mesh
    var mesh = new pc.Mesh();
    mesh.vertexBuffer = vertexBuffer;
    mesh.indexBuffer[0] = indexBuffer;
    mesh.primitive[0].type = pc.PRIMITIVE_TRIANGLES;
    mesh.primitive[0].base = 0;
    mesh.primitive[0].count = indices.length;
    mesh.primitive[0].indexed = true;

    // Calculer les bounding boxes (approximatives)
    mesh.aabb = new pc.BoundingBox();
    mesh.aabb.copy(this._computeAABB(meshData.positions));

    return mesh;
};


/**
 * _computeAABB(positions)
 * Calcule un bounding box pour un ensemble de points.
 */
MeshSlicer.prototype._computeAABB = function(positions) {
    var minx = +Infinity, miny = +Infinity, minz = +Infinity;
    var maxx = -Infinity, maxy = -Infinity, maxz = -Infinity;

    for (var i = 0; i < positions.length; i++) {
        var p = positions[i];
        if (p.x < minx) minx = p.x;
        if (p.y < miny) miny = p.y;
        if (p.z < minz) minz = p.z;

        if (p.x > maxx) maxx = p.x;
        if (p.y > maxy) maxy = p.y;
        if (p.z > maxz) maxz = p.z;
    }

    var center = new pc.Vec3(
        (minx + maxx) / 2,
        (miny + maxy) / 2,
        (minz + maxz) / 2
    );
    var halfExtents = new pc.Vec3(
        (maxx - minx) / 2,
        (maxy - miny) / 2,
        (maxz - minz) / 2
    );
    return new pc.BoundingBox(center, halfExtents);
};
