
/*

Timer class that uses playcanvas event loop.
Used for concurrency and animations.

Example:

MyScript.prototype.initialize = function() {
    const timer = this.createTimer({
        duration: 3,
        loop: true,
        active: true,
        onUpdate: (timer) => {
            // display current state
            const element = this.entity.element;
            element.text = `
                elapsed time: ${timer.elapsedTime}
                progress: ${timer.progress * 100}%
            `;
        },
        onEnd: (timer) => console.log('loop')
    });
};

*/

class Timer extends pc.EventHandler {
    /*

    Events:
    - update(timer)
    - end(timer)

    */

    constructor(script, {duration = 1, active = false,
        infinite = false, loop = false, onUpdate, onEnd}) {

        super();

        this._script = script;
        this._active = active;
        this._time = 0;
        this._delta = 0;
        this.duration = duration;
        this.infinite = infinite;
        this.loop = loop;
        
        if(onUpdate) this.on('update', onUpdate);
        if(onEnd) this.on('end', onEnd);

        script.app.on('update', this._update, this);
        script.once('destroy', this.destroy, this);
    }

    _update(dt) {
        if(!this._active) return;

        let ended = false;

        this._time += dt;
        this._delta = dt;

        if(!this.infinite && this._time >= this._duration) {
            if(this.loop) {
                this._time %= this._duration;
            } else {
                this._time = this._duration;
            }

            ended = true;
        }

        this._updateTween?.();
        this.fire('update', this);

        if(ended && this._active) {
            this._active = this.loop;
            this.fire('end', this);
        }
    }

    get active() {
        return this._active;
    }

    get elapsedTime() {
        return this._time;
    }

    get deltaTime() {
        return this._delta;
    }

    get progress() {
        return this._time / this._duration;
    }

    get duration() {
        return this._duration;
    }

    set duration(time) {
        this._duration = Math.max(0.001, time);
    }

    start() {
        this._active = true;
        this._time = 0;
    }

    pause() {
        this._active = false;
    }

    resume() {
        this._active = true;
    }

    destroy() {
        this._script.app.off('update', this._update, this);
    }
}

pc.ScriptType.prototype.createTimer = function(params) {
    return new Timer(this, params);
};
