// --------------- POST EFFECT DEFINITION --------------- //
/**
 * @class
 * @name GodRaysEffect
 * @classdesc PostEffect pour un halo + rayons de lumière, avec occlusion, pulsation, palette, et “spokes” plus marqués.
 * @augments pc.PostEffect
 */
function GodRaysEffect(graphicsDevice) {
    pc.PostEffect.call(this, graphicsDevice);

    // On veut accéder au buffer de profondeur
    this.needsDepthBuffer = true;

    // --- Vertex shader commun ---
    var commonVert = [
        graphicsDevice.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3VS : "",
        "attribute vec2 aPosition;",
        "varying vec2 vUv0;",
        "void main() {",
        "    gl_Position = vec4(aPosition, 0.0, 1.0);",
        "    vUv0 = (aPosition.xy + 1.0) * 0.5;",
        "}"
    ].join("\n");

    // --- Pass 1 : halo brut ---
    this.lightScatterShader = new pc.Shader(graphicsDevice, {
        attributes: { aPosition: pc.SEMANTIC_POSITION },
        vshader: commonVert,
        fshader: [
            graphicsDevice.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3PS : "",
            "precision " + graphicsDevice.precision + " float;",
            "",
            "varying vec2 vUv0;",
            "uniform sampler2D uColorBuffer;",
            "uniform float uAspect;",
            "uniform vec4  uLightPosition;", // (x, y, z, w)
            "uniform float uIntensity;",
            "uniform float uWeight;",
            "",
            "float sunHalo(vec2 uv, vec2 p) {",
            "    float di = distance(uv, p) * uLightPosition.w;",
            "    return (di <= 0.3333 / uWeight ? sqrt(1.0 - di * 3.0 / uWeight) : 0.0);",
            "}",
            "",
            "void main() {",
            "    vec2 uv = vUv0;",
            "    vec2 coords = uv; coords.x *= uAspect;",
            "    vec2 sunPos = uLightPosition.xy; sunPos.x *= uAspect;",
            "",
            "    float light = sunHalo(coords, sunPos);",
            "    float col = light * uIntensity;",
            "",
            "    // Stocke le halo dans R (si la lumière est derrière la cam => z=0, col=0)",
            "    gl_FragColor = vec4(col * uLightPosition.z, 0.0, 0.0, 1.0);",
            "}"
        ].join("\n")
    });

    // --- Pass 2 : Raymarch + occlusion + “spokes” (rayons accentués) ---
    this.blurShader = new pc.Shader(graphicsDevice, {
        attributes: { aPosition: pc.SEMANTIC_POSITION },
        vshader: commonVert,
        fshader: [
            graphicsDevice.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3PS : "",
            "precision " + graphicsDevice.precision + " float;",
            "",
            pc.shaderChunks.screenDepthPS, // pour getLinearScreenDepth(vUv)
            "",
            "#define MAX_SAMPLES 128",
            "varying vec2 vUv0;",
            "",
            "uniform sampler2D uColorBuffer;",         // Scène
            "uniform sampler2D uLightScatterBuffer;",  // Pass 1
            "uniform float uSamples;",                 // échantillons raymarch
            "uniform float uDecay;",                   // = 1.0 - userDecay
            "uniform float uExposure;",
            "uniform vec3  uColor;",
            "uniform float uPulseValue;",
            "uniform vec4  uLightPosition;",           // x, y, z=1 => devant, w=devicePixelRatio
            "uniform float uAspect;",
            "",
            "uniform float uOcclusionCutoff;",         // Si getLinearScreenDepth < cutoff => objet devant
            "",
            "// Spokes (rayons accentués)",
            "uniform float uNumSpokes;",    // Nombre de rayons
            "uniform float uSpokeIntensity;",  // Intensité du renforcement
            "uniform float uSpokeWidth;",      // Largeur du spoke
            "",
            "// Palette / Couleur alternée",
            "// => gérée côté script, on ne fait qu'appliquer uColor final",
            "",
            "// Simple random pour dithering",
            "float random(const vec2 uv) {",
            "    const float a = 12.9898, b = 78.233, c = 43758.5453;",
            "    float dt = dot(uv.xy, vec2(a, b));",
            "    float sn = mod(dt, 3.14159265359);",
            "    return fract(sin(sn) * c);",
            "}",
            "",
            "void main() {",
            "    vec2 uv = vUv0;",
            "    // Récupère le halo brut (canal R)",
            "    float occ = texture2D(uLightScatterBuffer, uv).r;",
            "    float dither = random(uv);",
            "",
            "    // Raymarch radial vers la source",
            "    vec2 coord = uv;",
            "    vec2 lightPos = uLightPosition.xy;",
            "    lightPos.x *= uAspect;",
            "    coord.x     *= uAspect;",
            "    vec2 dtc = (coord - lightPos) * (1.0 / uSamples * 0.95);",
            "    float illumDecay = 1.0;",
            "",
            "    for(int i=0; i<MAX_SAMPLES; i++) {",
            "        if(float(i)>=uSamples) break;",
            "        coord -= dtc;",
            "        float s = texture2D(uLightScatterBuffer, coord + dtc*dither).r;",
            "        s *= illumDecay * 0.25;  // weight ~ 0.25",
            "        occ += s;",
            "        illumDecay *= uDecay;",
            "    }",
            "",
            "    // Applique la pulsation calculée côté script",
            "    float rays = occ * uExposure * uLightPosition.z * uPulseValue;",
            "",
            "    // Lis la scène de base",
            "    vec4 base = texture2D(uColorBuffer, uv);",
            "",
            "    // Occlusion basique : si depth < cutoff => un objet masque la lumière",
            "    float sceneDepth = getLinearScreenDepth(vUv0);",
            "    if(uLightPosition.z>0.5 && sceneDepth < uOcclusionCutoff) {",
            "        rays = 0.0;",
            "    }",
            "",
            "    // --- Génération de spokes (rayons accentués) ---",
            "    vec2 off = uv;",
            "    off.x *= uAspect;",
            "    off -= lightPos;",
            "",
            "    // angle en [-PI..PI]",
            "    float angle = atan(off.y, off.x);",
            "    // on normalise en [0..1], puis on multiplie par le nombre de spokes",
            "    float normalizedAngle = fract((angle / 6.2831853) + 0.5) * uNumSpokes;",
            "",
            "    // on crée un masque (ligne fine) via smoothstep autour de fract(normalizedAngle)=0.5",
            "    float fractAngle = fract(normalizedAngle);",
            "    float lineMask = 1.0 - smoothstep(0.1 - uSpokeWidth, 0.1 + uSpokeWidth, fractAngle);",
            "",
            "    // spoke = intensité supplémentaire",
            "    float spokes = rays * lineMask * uSpokeIntensity;",
            "    float finalRays = rays + spokes;",
            "",
            "    // Couleur finale du halo",
            "    vec3 haloColor = uColor * finalRays;",
            "    // On additionne à la scène",
            "    gl_FragColor = base + vec4(haloColor, 1.0);",
            "}"
        ].join("\n")
    });

    // RT pour la passe 1
    var w = graphicsDevice.width, h = graphicsDevice.height;
    var cbuffer = new pc.Texture(graphicsDevice, {
        format: pc.PIXELFORMAT_R8_G8_B8_A8,
        width: w,
        height: h,
        mipmaps: false
    });
    cbuffer.minFilter = pc.FILTER_LINEAR;
    cbuffer.magFilter = pc.FILTER_LINEAR;
    cbuffer.addressU = pc.ADDRESS_CLAMP_TO_EDGE;
    cbuffer.addressV = pc.ADDRESS_CLAMP_TO_EDGE;

    this.targetLightScatter = new pc.RenderTarget(graphicsDevice, cbuffer, { depth: false });

    // Propriétés
    this.cameraEntity = null;
    this.lightEntity  = null;
    this.vec = new pc.Vec3();
    this.lightPosition = new pc.Vec4();

    // Paramètres generaux
    this.intensity = 1.5;
    this.weight    = 0.3;
    this.decay     = 0.01;
    this.exposure  = 0.09;
    this.color     = new pc.Color(1,1,1);

    this.samples   = 32;
    this.pulseValue = 1.0;

    // Occlusion
    this.occlusionCutoff = 0.9;

    // Spokes
    this.numSpokes       = 12.0;
    this.spokeIntensity  = 0.5;
    this.spokeWidth      = 0.02;  // 0.02 => trait fin
}
GodRaysEffect.prototype = Object.create(pc.PostEffect.prototype);
GodRaysEffect.prototype.constructor = GodRaysEffect;

Object.assign(GodRaysEffect.prototype, {
    render: function(inputTarget, outputTarget, rect) {
        var device = this.device;
        var scope  = device.scope;

        // Transforme la position 3D de la lumière en coord. écran
        if(this.cameraEntity && this.lightEntity) {
            var pos = this.lightEntity.getPosition();
            this.cameraEntity.camera.worldToScreen(pos, this.vec);

            var rx = device.width  / window.devicePixelRatio;
            var ry = device.height / window.devicePixelRatio;

            this.lightPosition.x = this.vec.x / rx;
            this.lightPosition.y = 1.0 - (this.vec.y / ry);
            // z=1 => devant la cam, 0 => derrière
            this.lightPosition.z = (this.vec.z > 0 ? 1.0 : 0.0);
            this.lightPosition.w = window.devicePixelRatio;
        }

        // Pass 1
        scope.resolve("uLightPosition").setValue([
            this.lightPosition.x, this.lightPosition.y,
            this.lightPosition.z, this.lightPosition.w
        ]);
        scope.resolve("uAspect").setValue(device.width/device.height);
        scope.resolve("uColorBuffer").setValue(inputTarget.colorBuffer);
        scope.resolve("uIntensity").setValue(this.intensity);
        scope.resolve("uWeight").setValue(this.weight);

        pc.drawFullscreenQuad(device, this.targetLightScatter, this.vertexBuffer, this.lightScatterShader, rect);

        // Pass 2
        scope.resolve("uLightScatterBuffer").setValue(this.targetLightScatter.colorBuffer);
        scope.resolve("uColorBuffer").setValue(inputTarget.colorBuffer);

        scope.resolve("uSamples").setValue(this.samples);
        scope.resolve("uDecay").setValue(1.0 - this.decay);
        scope.resolve("uExposure").setValue(this.exposure);
        scope.resolve("uColor").setValue([this.color.r, this.color.g, this.color.b]);
        scope.resolve("uPulseValue").setValue(this.pulseValue);

        scope.resolve("uOcclusionCutoff").setValue(this.occlusionCutoff);

        // Spokes
        scope.resolve("uNumSpokes").setValue(this.numSpokes);
        scope.resolve("uSpokeIntensity").setValue(this.spokeIntensity);
        scope.resolve("uSpokeWidth").setValue(this.spokeWidth);

        scope.resolve("uLightPosition").setValue([
            this.lightPosition.x, this.lightPosition.y,
            this.lightPosition.z, this.lightPosition.w
        ]);
        scope.resolve("uAspect").setValue(device.width/device.height);

        pc.drawFullscreenQuad(device, outputTarget, this.vertexBuffer, this.blurShader, rect);
    }
});

// ----------------- SCRIPT DEFINITION ------------------ //
var GodRays = pc.createScript("godRays");

// Attributs

GodRays.attributes.add("cameraEntity", { type:"entity" });
GodRays.attributes.add("lightEntity",  { type:"entity" });

GodRays.attributes.add("samples",    { type:"number", default:32, min:1, max:128 });
GodRays.attributes.add("intensity",  { type:"number", default:1.5, min:0, max:5 });
GodRays.attributes.add("weight",     { type:"number", default:0.3, min:0, max:1 });
GodRays.attributes.add("decay",      { type:"number", default:0.01, precision:3, min:0, max:1 });
GodRays.attributes.add("exposure",   { type:"number", default:0.09, min:0, max:1 });
GodRays.attributes.add("color",      { type:"rgb", default:[1,1,1] });

// Pulsation
GodRays.attributes.add("enablePulse", {
    type:"boolean",
    default:true,
    title:"Enable Pulse"
});
GodRays.attributes.add("pulseMode", {
    type:"string",
    default:"sin",
    enum:[
        {"Sinus":"sin"},
        {"Triangle":"triangle"},
        {"Heart":"heart"}
    ],
    title:"Pulse Mode"
});
GodRays.attributes.add("pulseSpeed", {
    type:"number",
    default:2.0,
    title:"Pulse Speed"
});
GodRays.attributes.add("pulseAmplitude", {
    type:"number",
    default:0.5,
    min:0, max:1, precision:2,
    title:"Pulse Amplitude"
});

// Palette (optionnel)
GodRays.attributes.add("useColorPalette", {
    type:"boolean",
    default:false,
    title:"Use Color Palette"
});
GodRays.attributes.add("colorPalette", {
    type:"string",
    default:"[[1,0,0],[0,1,0],[0,0,1]]",
    title:"Color Palette (JSON)"
});
GodRays.attributes.add("paletteCycleSpeed", {
    type:"number",
    default:0.5,
    title:"Palette Cycle Speed"
});

// Occlusion simple
GodRays.attributes.add("occlusionCutoff", {
    type:"number",
    default:0.9,
    min:0, max:1,
    title:"Occlusion Cutoff"
});

// Spokes (rayons accentués)
GodRays.attributes.add("numSpokes", {
    type:"number",
    default:8,
    min:1, max:64,
    precision:0,
    title:"Number of spokes"
});
GodRays.attributes.add("spokeIntensity", {
    type:"number",
    default:0.5,
    min:0, max:5,
    title:"Spokes Intensity"
});
GodRays.attributes.add("spokeWidth", {
    type:"number",
    default:0.02,
    precision:3,
    min:0, max:0.5,
    title:"Spoke Width"
});

// Implementation

GodRays.prototype.initialize = function() {
    if(!this.cameraEntity) return;

    var queue = this.cameraEntity.camera.postEffects;
    this.effect = new GodRaysEffect(this.app.graphicsDevice);

    this.syncAttributesToEffect();
    queue.addEffect(this.effect);

    // Sur changement d'attribut
    this.on("attr", function(name, value) {
        // S'il s'agit de samples, on recrée l'effet
        if(name==="samples") {
            queue.removeEffect(this.effect);
            this.effect = new GodRaysEffect(this.app.graphicsDevice);
        }
        this.syncAttributesToEffect();

        if(this.enabled) {
            queue.addEffect(this.effect);
        }
    }, this);

    // (Dés)activation
    this.on("state", function(enabled) {
        if(enabled) {
            queue.addEffect(this.effect);
        } else {
            queue.removeEffect(this.effect);
        }
    }, this);

    // Sur destruction
    this.on("destroy", function() {
        queue.removeEffect(this.effect);
    });
};

GodRays.prototype.update = function(dt) {
    // Calcul de la pulsation + palette

    // 1) Avance un temps interne (pour la palette)
    this.effect._time = (this.effect._time || 0) + dt;

    // 2) Palette
    var finalColor = new pc.Color(this.color.r, this.color.g, this.color.b);
    if(this.useColorPalette && this._parsedPalette && this._parsedPalette.length>0) {
        var t = (this.effect._time || 0) * this.paletteCycleSpeed;
        var idx = Math.floor(t) % this._parsedPalette.length;
        var cArr = this._parsedPalette[idx];
        if(cArr && cArr.length===3) {
            finalColor.set(cArr[0], cArr[1], cArr[2]);
        }
    }
    this.effect.color = finalColor;

    // 3) Pulsation
    var pulse = 1.0;
    if(this.enablePulse) {
        var timeScaled = (this.effect._time || 0) * this.pulseSpeed;
        switch(this.pulseMode) {
            case "triangle":
                var frac = timeScaled % 1.0;
                var tri  = frac*2.0;
                if(tri>1.0) tri=2.0-tri;
                tri   = 2.0*tri - 1.0; // => -1..+1
                pulse = 1.0 + this.pulseAmplitude * tri;
                break;
            case "heart":
                var beat = Math.sin(timeScaled)*Math.sin(timeScaled*0.5);
                pulse = 1.0 + this.pulseAmplitude * beat;
                break;
            case "sin":
            default:
                pulse = 1.0 + this.pulseAmplitude * Math.sin(timeScaled);
                break;
        }
    }
    this.effect.pulseValue = pulse;
};

GodRays.prototype.syncAttributesToEffect = function() {
    var e = this.effect;

    // Liens entités
    e.cameraEntity = this.cameraEntity;
    e.lightEntity  = this.lightEntity;

    // Paramètres
    e.samples      = this.samples;
    e.intensity    = this.intensity;
    e.weight       = this.weight;
    e.decay        = this.decay;
    e.exposure     = this.exposure;
    e.color.set(this.color.r, this.color.g, this.color.b);

    // Occlusion
    e.occlusionCutoff = this.occlusionCutoff;

    // Spokes
    e.numSpokes       = this.numSpokes;
    e.spokeIntensity  = this.spokeIntensity;
    e.spokeWidth      = this.spokeWidth;

    // Palette
    this._parsedPalette = [];
    if(this.useColorPalette && this.colorPalette) {
        try {
            this._parsedPalette = JSON.parse(this.colorPalette);
        } catch(err) {
            console.warn("Invalid JSON in colorPalette:", err);
        }
    }
};
