// --------------- POST EFFECT DEFINITION --------------- //
/**
 * @class
 * @name LensFlareEffect
 * @classdesc PostEffect pour un effet de lens flare amélioré, avec extraction des hautes lumières, ghost images, débordement de rayons et cercles lumineux.
 * @augments pc.PostEffect
 */
function LensFlareEffect(graphicsDevice) {
    pc.PostEffect.call(this, graphicsDevice);
    
    // On souhaite utiliser la depth buffer pour éventuellement gérer l'occlusion
    this.needsDepthBuffer = true;

    // --- Vertex shader commun ---
    var commonVert = [
        graphicsDevice.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3VS : "",
        "attribute vec2 aPosition;",
        "varying vec2 vUv0;",
        "void main() {",
        "    gl_Position = vec4(aPosition, 0.0, 1.0);",
        "    vUv0 = (aPosition.xy + 1.0) * 0.5;",
        "}"
    ].join("\n");

    // --- Pass 1 : Extraction des hautes lumières (bright pass) ---
    this.brightPassShader = new pc.Shader(graphicsDevice, {
        attributes: { aPosition: pc.SEMANTIC_POSITION },
        vshader: commonVert,
        fshader: [
            graphicsDevice.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3PS : "",
            "precision " + graphicsDevice.precision + " float;",
            "",
            "varying vec2 vUv0;",
            "uniform sampler2D uColorBuffer;", // Scène
            "uniform float uThreshold;",       // Seuil de luminosité
            "uniform float uBrightFactor;",    // Facteur d'amplification
            "",
            "void main() {",
            "    vec4 color = texture2D(uColorBuffer, vUv0);",
            "    float brightness = max(color.r, max(color.g, color.b));",
            "    if(brightness > uThreshold) {",
            "        gl_FragColor = color * (brightness - uThreshold) * uBrightFactor;",
            "    } else {",
            "        gl_FragColor = vec4(0.0);",
            "    }",
            "}"
        ].join("\n")
    });

    // --- Pass 2 : Génération du lens flare amélioré (ghost images, débordement et cercles lumineux) ---
    this.lensFlareShader = new pc.Shader(graphicsDevice, {
        attributes: { aPosition: pc.SEMANTIC_POSITION },
        vshader: commonVert,
        fshader: [
            // Version en GLSL selon WebGL version
            graphicsDevice.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3PS : "",
            "precision " + graphicsDevice.precision + " float;",
            "",
            "varying vec2 vUv0;",
            "uniform sampler2D uColorBuffer;",    // Scène de base
            "uniform sampler2D uBrightPass;",     // Texture issue de la passe 1",
            "uniform vec4 uLightPosition;",       // (x, y, visibilité, pixelRatio)",
            "uniform float uNumGhosts;",          // Nombre de ghost images",
            "uniform float uGhostSpacing;",       // Espacement entre ghost images",
            "uniform float uGhostIntensity;",     // Intensité de base des ghost images",
            "",
            "// Nouveaux paramètres pour personnaliser l'effet",
            "uniform vec3  uFlareColor;",          // Couleur choisie pour l'effet",
            "uniform float uEffectAccentuation;",  // Facteur pour accentuer l'effet",
            "uniform float uFlareRadius;",         // Rayon des cercles lumineux",
            "",
            "void main() {",
            "    vec2 uv = vUv0;",
            "    vec4 base = texture2D(uColorBuffer, uv);",
            "",
            "    // Si la source lumineuse n'est pas visible, on ne calcule pas le lens flare",
            "    if(uLightPosition.z < 0.5) {",
            "        gl_FragColor = base;",
            "        return;",
            "    }",
            "",
            "    vec2 center = vec2(0.5, 0.5);",
            "    vec2 dir = center - uLightPosition.xy;",
            "    vec3 flare = vec3(0.0);",
            "",
            "    #define MAX_GHOSTS 5",
            "    for (int i = 0; i < MAX_GHOSTS; i++) {",
            "        if(float(i) >= uNumGhosts) break;",
            "        float factor = (float(i) + 1.0) * uGhostSpacing;",
            "        vec2 ghostPos = uLightPosition.xy + dir * factor;",
            "        vec3 ghost = texture2D(uBrightPass, ghostPos).rgb;",
            "",
            "        // Application de l'intensité et de l'accentuation, avec décroissance pour chaque ghost",
            "        ghost *= uGhostIntensity / (float(i) + 1.0) * uEffectAccentuation;",
            "",
            "        // Calcul d'un masque circulaire pour créer des cercles lumineux autour de chaque ghost",
            "        float dist = distance(uv, ghostPos);",
            "        float circle = 1.0 - smoothstep(uFlareRadius * 0.8, uFlareRadius, dist);",
            "",
            "        flare += ghost * (1.0 + circle);",
            "    }",
            "",
            "    // Ajoute un halo central à la position de la lumière",
            "    vec3 halo = texture2D(uBrightPass, uLightPosition.xy).rgb;",
            "    float distCenter = distance(uv, uLightPosition.xy);",
            "    float circleCenter = 1.0 - smoothstep(uFlareRadius * 0.8, uFlareRadius, distCenter);",
            "    halo *= uGhostIntensity * uEffectAccentuation;",
            "    halo *= (1.0 + circleCenter);",
            "    flare += halo;",
            "",
            "    // Applique la couleur choisie sur l'ensemble du flare",
            "    flare *= uFlareColor;",
            "",
            "    vec3 finalColor = base.rgb + flare;",
            "    gl_FragColor = vec4(finalColor, base.a);",
            "}"
        ].join("\n")
    });

    // Création de la RT pour la passe 1 (bright pass)
    var w = graphicsDevice.width, h = graphicsDevice.height;
    var brightTexture = new pc.Texture(graphicsDevice, {
        format: pc.PIXELFORMAT_R8_G8_B8_A8,
        width: w,
        height: h,
        mipmaps: false
    });
    brightTexture.minFilter = pc.FILTER_LINEAR;
    brightTexture.magFilter = pc.FILTER_LINEAR;
    brightTexture.addressU = pc.ADDRESS_CLAMP_TO_EDGE;
    brightTexture.addressV = pc.ADDRESS_CLAMP_TO_EDGE;

    this.targetBrightPass = new pc.RenderTarget(graphicsDevice, brightTexture, { depth: false });

    // Propriétés de base
    this.cameraEntity = null;
    this.lightEntity  = null;
    this.vec = new pc.Vec3();
    this.lightPosition = new pc.Vec4();

    // Paramètres pour le bright pass
    this.threshold = 0.8;
    this.brightFactor = 1.0;

    // Paramètres pour le lens flare (ghost images)
    this.numGhosts = 4.0;       // Nombre de ghost images (max 5)
    this.ghostSpacing = 0.25;   // Espacement relatif
    this.ghostIntensity = 1.0;  // Intensité de base des ghost images

    // Nouveaux paramètres pour personnaliser l'effet
    this.flareColor = new pc.Color(1, 1, 1);
    this.effectAccentuation = 1.0;
    this.flareRadius = 0.1;     // Ajuste le rayon des cercles lumineux
}
LensFlareEffect.prototype = Object.create(pc.PostEffect.prototype);
LensFlareEffect.prototype.constructor = LensFlareEffect;

Object.assign(LensFlareEffect.prototype, {
    render: function(inputTarget, outputTarget, rect) {
        var device = this.device;
        var scope  = device.scope;

        // Transforme la position 3D de la lumière en coordonnées écran
        if (this.cameraEntity && this.lightEntity) {
            var pos = this.lightEntity.getPosition();
            this.cameraEntity.camera.worldToScreen(pos, this.vec);

            var rx = device.width  / window.devicePixelRatio;
            var ry = device.height / window.devicePixelRatio;

            this.lightPosition.x = this.vec.x / rx;
            this.lightPosition.y = 1.0 - (this.vec.y / ry);
            // z = 1 si la lumière est devant la caméra, 0 sinon
            this.lightPosition.z = (this.vec.z > 0 ? 1.0 : 0.0);
            this.lightPosition.w = window.devicePixelRatio;
        }

        // Passe 1 : extraction des hautes lumières (bright pass)
        scope.resolve("uColorBuffer").setValue(inputTarget.colorBuffer);
        scope.resolve("uThreshold").setValue(this.threshold);
        scope.resolve("uBrightFactor").setValue(this.brightFactor);
        pc.drawFullscreenQuad(device, this.targetBrightPass, this.vertexBuffer, this.brightPassShader, rect);

        // Passe 2 : génération du lens flare amélioré
        scope.resolve("uColorBuffer").setValue(inputTarget.colorBuffer);
        scope.resolve("uBrightPass").setValue(this.targetBrightPass.colorBuffer);
        scope.resolve("uLightPosition").setValue([
            this.lightPosition.x, this.lightPosition.y,
            this.lightPosition.z, this.lightPosition.w
        ]);
        scope.resolve("uNumGhosts").setValue(this.numGhosts);
        scope.resolve("uGhostSpacing").setValue(this.ghostSpacing);
        scope.resolve("uGhostIntensity").setValue(this.ghostIntensity);

        // Nouveaux uniforms pour la personnalisation
        scope.resolve("uFlareColor").setValue([this.flareColor.r, this.flareColor.g, this.flareColor.b]);
        scope.resolve("uEffectAccentuation").setValue(this.effectAccentuation);
        scope.resolve("uFlareRadius").setValue(this.flareRadius);

        pc.drawFullscreenQuad(device, outputTarget, this.vertexBuffer, this.lensFlareShader, rect);
    }
});

// ----------------- SCRIPT DEFINITION ------------------ //
var LensFlare = pc.createScript("lensFlare");

// Attributs de base
LensFlare.attributes.add("cameraEntity", { type:"entity" });
LensFlare.attributes.add("lightEntity",  { type:"entity" });

LensFlare.attributes.add("threshold",    { type:"number", default:0.8, min:0, max:1, title:"Threshold" });
LensFlare.attributes.add("brightFactor", { type:"number", default:1.0, min:0, max:5, title:"Bright Factor" });

LensFlare.attributes.add("numGhosts",       { type:"number", default:4, min:1, max:5, precision:0, title:"Number of Ghosts" });
LensFlare.attributes.add("ghostSpacing",    { type:"number", default:0.25, min:0, max:1, title:"Ghost Spacing" });
LensFlare.attributes.add("ghostIntensity",  { type:"number", default:1.0, min:0, max:5, title:"Ghost Intensity" });

// Nouveaux attributs pour l'accentuation et la couleur
LensFlare.attributes.add("flareColor", {
    type: "rgb",
    default: [1, 1, 1],
    title: "Flare Color"
});
LensFlare.attributes.add("effectAccentuation", {
    type: "number",
    default: 1.0,
    min: 0,
    max: 5,
    title: "Effect Accentuation"
});
LensFlare.attributes.add("flareRadius", {
    type: "number",
    default: 0.1,
    min: 0,
    max: 0.5,
    title: "Flare Radius"
});

LensFlare.prototype.initialize = function() {
    if (!this.cameraEntity) return;

    var queue = this.cameraEntity.camera.postEffects;
    this.effect = new LensFlareEffect(this.app.graphicsDevice);

    this.syncAttributesToEffect();
    queue.addEffect(this.effect);

    // Sur modification d'attribut, on synchronise l'effet
    this.on("attr", function(name, value) {
        this.syncAttributesToEffect();
    }, this);

    // (Dés)activation
    this.on("state", function(enabled) {
        if (enabled) {
            queue.addEffect(this.effect);
        } else {
            queue.removeEffect(this.effect);
        }
    }, this);

    // À la destruction
    this.on("destroy", function() {
        queue.removeEffect(this.effect);
    });
};

LensFlare.prototype.syncAttributesToEffect = function() {
    var e = this.effect;
    e.cameraEntity = this.cameraEntity;
    e.lightEntity  = this.lightEntity;

    e.threshold = this.threshold;
    e.brightFactor = this.brightFactor;
    e.numGhosts = this.numGhosts;
    e.ghostSpacing = this.ghostSpacing;
    e.ghostIntensity = this.ghostIntensity;

    e.flareColor = new pc.Color(this.flareColor.r, this.flareColor.g, this.flareColor.b);
    e.effectAccentuation = this.effectAccentuation;
    e.flareRadius = this.flareRadius;
};
