var FlameHalo = pc.createScript("flameHalo");

// Attributs pour l'effet (à paramétrer dans l'éditeur)
FlameHalo.attributes.add("cameraEntity", { type: "entity", title: "Camera Entity" });
FlameHalo.attributes.add("haloColor", { type: "rgb", default: [1, 0.5, 0], title: "Halo Color" });
FlameHalo.attributes.add("haloScale", { type: "number", default: 2.0, title: "Halo Scale" });
FlameHalo.attributes.add("intensity", { type: "number", default: 1.0, title: "Intensity", min: 0, max: 5 });
FlameHalo.attributes.add("flickerSpeed", { type: "number", default: 10.0, title: "Flicker Speed", min: 0, max: 50 });

// ------------------ INITIALISATION ------------------ //
FlameHalo.prototype.initialize = function () {
    this.time = 0;
    
    // Création d'une entité enfant qui portera l'effet de halo
    this.haloEntity = new pc.Entity("FlameHalo");
    this.haloEntity.addComponent("model", {
        type: "plane"
    });
    
    // Création d'un matériau personnalisé avec shader pour le halo
    var device = this.app.graphicsDevice;
    
    // Vertex shader : passe les coordonnées UV au fragment shader
    var vshader = [
        (device.webgl2 ? "#version 300 es\n" : ""),
        "attribute vec3 aPosition;",
        "attribute vec2 aUv0;",
        "varying vec2 vUv0;",
        "uniform mat4 matrix_model;",
        "uniform mat4 matrix_viewProjection;",
        "void main(void) {",
        "    vUv0 = aUv0;",
        "    gl_Position = matrix_viewProjection * matrix_model * vec4(aPosition, 1.0);",
        "}"
    ].join("\n");
    
    // Fragment shader : génère un dégradé radial et y ajoute une modulation de flicker
    var fshader = [
        (device.webgl2 ? "#version 300 es\nprecision mediump float;\n" : "precision mediump float;"),
        "varying vec2 vUv0;",
        "uniform vec3 uHaloColor;",
        "uniform float uIntensity;",
        "uniform float uTime;",
        "float random(vec2 st) {",
        "    return fract(sin(dot(st.xy, vec2(12.9898,78.233))) * 43758.5453123);",
        "}",
        "void main(void) {",
        "    vec2 uv = vUv0;",
        "    // Calcul de la distance du centre (0.5, 0.5) du plan",
        "    float dist = distance(uv, vec2(0.5, 0.5));",
        "    // Génère un gradient radial : le halo est le plus intense au centre",
        "    float alpha = smoothstep(0.5, 0.0, dist);",
        "    // Ajout d'un effet de pulsation (flicker)",
        "    float flicker = 0.8 + 0.2 * sin(uTime * " + this.flickerSpeed + ".0 + random(uv) * 6.28);",
        "    alpha *= flicker * uIntensity;",
        "    gl_FragColor = vec4(uHaloColor, alpha);",
        "}"
    ].join("\n");
    
    // Création du shader personnalisé
    var shader = new pc.Shader(device, {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION,
            aUv0: pc.SEMANTIC_TEXCOORD0
        },
        vshader: vshader,
        fshader: fshader
    });
    
    // Création du matériau et configuration du blending additif
    this.haloMaterial = new pc.Material();
    this.haloMaterial.shader = shader;
    this.haloMaterial.blendType = pc.BLEND_ADDITIVE;
    this.haloMaterial.depthWrite = false;
    this.haloMaterial.depthTest = false;
    this.haloMaterial.update();
    
    // Affecte le matériau personnalisé au modèle du halo
    this.haloEntity.model.material = this.haloMaterial;
    
    // Positionne le halo au centre de l'entité parente et ajuste son échelle
    this.haloEntity.setLocalPosition(0, 0, 0);
    this.haloEntity.setLocalScale(this.haloScale, this.haloScale, this.haloScale);
    
    // Ajoute le halo en tant qu'enfant de l'entité à laquelle ce script est attaché
    this.entity.addChild(this.haloEntity);
};

// ------------------ UPDATE ------------------ //
FlameHalo.prototype.update = function (dt) {
    this.time += dt;
    
    // Mise à jour des paramètres du shader (temps, couleur, intensité)
    this.haloMaterial.setParameter("uTime", this.time);
    this.haloMaterial.setParameter("uHaloColor", [this.haloColor.r, this.haloColor.g, this.haloColor.b]);
    this.haloMaterial.setParameter("uIntensity", this.intensity);
    
    // Le halo doit toujours faire face à la caméra pour un effet billboard
    if (this.cameraEntity) {
        this.haloEntity.lookAt(this.cameraEntity.getPosition());
    }
};
