// --------------- POST EFFECT DEFINITION --------------- //
/**
 * @class
 * @name PortalTransitionEffect
 * @classdesc PostEffect gérant un effet de “portail lumineux” 
 * qui inonde progressivement l'écran, avec un halo hexagonal 
 * et des reflets arc-en-ciel.
 * @augments pc.PostEffect
 */

function PortalTransitionEffect(graphicsDevice) {
    pc.PostEffect.call(this, graphicsDevice);

    // On n'a pas besoin du depth buffer pour cette transition :
    this.needsDepthBuffer = false;

    // Paramètres exposés (on les mettra à jour via le script PortalTransition)
    this.progress    = 0.0;           // 0..1 => état de la transition
    this.portalColor = new pc.Color(1, 1, 1); // Couleur centrale du portail
    this.rainbowIntensity = 1.0;      // Intensité de l’iridescence (arc-en-ciel)
    this.hexSize     = 0.2;           // “Taille” initiale de l'hexagone
    this.fadePower   = 2.0;           // Contrôle la vitesse à laquelle l’écran devient blanc
    this.center      = new pc.Vec2(0.5, 0.5); // Centre du portail (UV)
    
    // Vertex shader de base (full-screen quad)
    var commonVert = [
        graphicsDevice.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3VS : "",
        "attribute vec2 aPosition;",
        "varying vec2 vUv0;",
        "void main(void) {",
        "    gl_Position = vec4(aPosition, 0.0, 1.0);",
        "    vUv0 = (aPosition.xy + 1.0) * 0.5;",
        "}"
    ].join("\n");

    // Fragment shader
    // On calcule une forme hexagonale qui grandit avec progress,
    // + un effet arc-en-ciel sur le bord, 
    // + inondation progressive de blanc.
    this.portalShader = new pc.Shader(graphicsDevice, {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION
        },
        vshader: commonVert,
        fshader: [
            graphicsDevice.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3PS : "",
            "precision " + graphicsDevice.precision + " float;",
            "",
            // Paramètre d’entrée
            "varying vec2 vUv0;",
            "uniform sampler2D uSceneTex;",
            "",
            "uniform float uProgress;",         // 0..1
            "uniform vec3  uPortalColor;",      // Couleur du portail
            "uniform float uRainbowIntensity;", // Intensité de l'effet arc-en-ciel
            "uniform float uHexSize;",          // Taille “initiale” du portail (hex)
            "uniform float uFadePower;",        // Contrôle la rapidité du passage au blanc
            "uniform vec2  uCenter;",           // Centre du portail (UV)
            "",
            "// Fonction pour calculer la distance à un hexagone centré en (0,0)",
            "// Inspirée de différents ‘Hex Dist’ tricks en GLSL",
            "float hexDist(vec2 p) {",
            "    // On ramène p dans [-1..+1], pour un hex plus facile à manipuler",
            "    // mais on peut aussi le garder en [0..1].",
            "    // Ici, p est déjà décalé par uCenter plus bas dans le main().",
            "",
            "    // Rotation de 45° pour aligner un hex ? Optionnel, ou 30°.",
            "    // On va plutôt s’inspirer d’un hex aligné sur l’axe vertical.",
            "",
            "    p = abs(p);",
            "    return max(p.x * 0.8660254 + p.y * 0.5, p.y);",
            "}",
            "",
            "// Petitte fonction pour un arc-en-ciel :",
            "// On utilise un angle => renvoie R, G, B ‘arc en ciel’.",
            "vec3 rainbowColor(float angle) {",
            "    // Décalages de 2π/3 = ~2.094 pour répartir R, G, B",
            "    float r = sin(angle) * 0.5 + 0.5;",
            "    float g = sin(angle + 2.094) * 0.5 + 0.5;",
            "    float b = sin(angle + 4.188) * 0.5 + 0.5;",
            "    return vec3(r, g, b);",
            "}",
            "",
            "void main() {",
            "    // Lis la scène de base",
            "    vec4 sceneColor = texture2D(uSceneTex, vUv0);",
            "",
            "    // Calcule coords relatifs au centre du portail",
            "    vec2 uv = vUv0 - uCenter;",
            "    // On scale un peu si besoin, pour régler la taille globale",
            "    // Ici, uHexSize sert d’échelle initiale, et uProgress fait grandir le rayon.",
            "    float scale = mix(uHexSize, 2.0, uProgress); // 2.0 => max covering screen",
            "    uv /= scale;",
            "",
            "    // Distance au contour de l’hexagone centré en (0,0)",
            "    float d = hexDist(uv);",
            "",
            "    // On considère qu’en dessous de d<1.0, on est “à l’intérieur” du portail.",
            "    // Mais on fait une transition progressive.",
            "    // => On peut utiliser un smoothstep autour de 1.0 pour avoir un bord flou.",
            "    float border = smoothstep(0.95, 1.0, d); // 0 = inside, 1 = outside",
            "",
            "    // Couleur du portail = blanc + nuance arc-en-ciel près du bord",
            "    // On veut un ‘anneau’ arc-en-ciel quand d ~ 1.0.",
            "    float ringMask = 1.0 - border; // 1.0 => inside, 0 => outside",
            "    float angle = atan(uv.y, uv.x); // angle en [-π..π]",
            "    vec3 rainbow = rainbowColor(angle) * uRainbowIntensity;",
            "    // On mixe la couleur blanche (uPortalColor) et l’arc-en-ciel proportionnellement au ringMask",
            "    float ringFactor = smoothstep(0.8, 1.0, d); // renforce le rainbow près du bord",
            "    vec3 portalCol = mix(uPortalColor, rainbow, ringFactor);",
            "",
            "    // On détermine combien le portail recouvre la scène",
            "    // => plus uProgress est grand, plus le portail recouvre tout (même en dehors de l’hex)",
            "    // => On fait un blend global via progress^fadePower",
            "    float globalWhite = pow(uProgress, uFadePower); // 0..1, exponentiel",
            "",
            "    // Couleur finale = la scène + la portion inside portal (1 - border)",
            "    // On va faire un alpha = ringMask, et un alpha global = globalWhite ?",
            "    // Ici on va combiner 2 choses :",
            "    // 1) La zone inside l’hex => on la dessine avec portalCol (transition).",
            "    // 2) Un fade global vers le blanc complet quand progress ~1.",
            "",
            "    // Couleur du pixel inside l’hex",
            "    vec3 inside = mix(sceneColor.rgb, portalCol, ringMask);",
            "",
            "    // Ensuite on mix la result (inside) avec le blanc total, piloté par globalWhite.",
            "    vec3 finalColor = mix(inside, vec3(1.0), globalWhite);",
            "",
            "    gl_FragColor = vec4(finalColor, 1.0);",
            "}"
        ].join("\n")
    });
}
PortalTransitionEffect.prototype = Object.create(pc.PostEffect.prototype);
PortalTransitionEffect.prototype.constructor = PortalTransitionEffect;

Object.assign(PortalTransitionEffect.prototype, {
    render: function (inputTarget, outputTarget, rect) {
        var device = this.device;
        var scope  = device.scope;

        // Assigne les uniforms
        scope.resolve("uSceneTex").setValue(inputTarget.colorBuffer);

        scope.resolve("uProgress").setValue(this.progress);
        scope.resolve("uPortalColor").setValue([this.portalColor.r, this.portalColor.g, this.portalColor.b]);
        scope.resolve("uRainbowIntensity").setValue(this.rainbowIntensity);
        scope.resolve("uHexSize").setValue(this.hexSize);
        scope.resolve("uFadePower").setValue(this.fadePower);
        scope.resolve("uCenter").setValue([this.center.x, this.center.y]);

        pc.drawFullscreenQuad(device, outputTarget, this.vertexBuffer, this.portalShader, rect);
    }
});


// ----------------- SCRIPT DE TRANSITION ------------------ //
var PortalTransition = pc.createScript("portalTransition");

/*
  Ce script:
  1) Crée une instance du post-effet PortalTransitionEffect
  2) Fait évoluer la variable "progress" de 0..1 en "duration" secondes
  3) Quand progress atteint 1, on peut charger la nouvelle scène, ou faire une suite
*/

// --- Paramètres
PortalTransition.attributes.add("cameraEntity", { 
    type: "entity", 
    title: "Camera",
    description: "Caméra sur laquelle on applique l'effet de portail"
});

PortalTransition.attributes.add("duration", {
    type: "number",
    default: 3,
    title: "Duration (seconds)"
});

PortalTransition.attributes.add("portalColor", {
    type: "rgb",
    default: [1, 1, 1],
    title: "Portal Color"
});

PortalTransition.attributes.add("rainbowIntensity", {
    type: "number",
    default: 1.0,
    min: 0, max: 5,
    title: "Rainbow Intensity"
});

PortalTransition.attributes.add("hexSize", {
    type: "number",
    default: 0.2,
    min: 0.01, max: 1,
    title: "Hex Start Size"
});

PortalTransition.attributes.add("fadePower", {
    type: "number",
    default: 2,
    min: 1, max: 5,
    title: "Fade Power"
});

PortalTransition.attributes.add("center", {
    type: "vec2",
    default: [0.5, 0.5],
    title: "Portal Center (UV)"
});

PortalTransition.prototype.initialize = function() {
    if (!this.cameraEntity || !this.cameraEntity.camera) {
        console.warn("PortalTransition: cameraEntity invalide ou pas de camera component");
        return;
    }

    // Crée l'instance du post-effect
    var device = this.app.graphicsDevice;
    this.effect = new PortalTransitionEffect(device);

    // Ajoute le postEffect à la caméra
    this.cameraEntity.camera.postEffects.addEffect(this.effect);

    // Synchronise les propriétés
    this.effect.portalColor.set(this.portalColor.r, this.portalColor.g, this.portalColor.b);
    this.effect.rainbowIntensity = this.rainbowIntensity;
    this.effect.hexSize = this.hexSize;
    this.effect.fadePower = this.fadePower;
    this.effect.center.set(this.center[0], this.center[1]);

    // On va animer "progress" de 0..1 sur "duration"
    this._time = 0;
    this._transitionDone = false;
};

PortalTransition.prototype.update = function(dt) {
    if (this._transitionDone) return;

    this._time += dt;
    var ratio = pc.math.clamp(this._time / this.duration, 0, 1);

    // Assigne la progress au post-effect
    this.effect.progress = ratio;

    if (ratio >= 1) {
        this._transitionDone = true;
        // Ici, tu peux charger la nouvelle scène, ou déclencher un événement, etc.
        // Ex :
        // this.app.scenes.loadScene("myNextScene.json");
    }
};

// Si on désactive ou détruit le script
PortalTransition.prototype.onDisable = function() {
    if (this.cameraEntity && this.cameraEntity.camera) {
        this.cameraEntity.camera.postEffects.removeEffect(this.effect);
    }
};

PortalTransition.prototype.onDestroy = function() {
    if (this.cameraEntity && this.cameraEntity.camera) {
        this.cameraEntity.camera.postEffects.removeEffect(this.effect);
    }
};
