/* ============================================================================
 * ShowOverlay.js
 * ----------------------------------------------------------------------------
 * Affiche une image PNG en overlay avec :
 *   – positionnement fiable (top‑left/right, bottom‑left/right, centered, stretched)
 *   – durée d’affichage réglable (0 = illimité)
 *   – fade‑in / fade‑out optionnels (250 ms)
 *   – rotation optionnelle
 * ----------------------------------------------------------------------------
 * Nécessite le script Tween (Tween.js) de PlayCanvas dans la scène
 * https://github.com/playcanvas/playcanvas-tween
 * ========================================================================== */

var ShowOverlay = pc.createScript('showOverlay');

/* ----------------------- ATTRIBUTS DE L’INSPECTEUR ----------------------- */

ShowOverlay.attributes.add('imageAsset',  { type: 'asset', assetType: 'texture', title: 'Image PNG' });
ShowOverlay.attributes.add('showEvent',   { type: 'string', default: 'AfficherImage', title: 'Événement d\'affichage' });
ShowOverlay.attributes.add('stopEvent',   { type: 'string', default: '', title: 'Événement d\'arrêt (optionnel)' });

ShowOverlay.attributes.add('position', {
    type: 'string',
    enum: [
        { 'Top Left':    'top-left'    },
        { 'Top Right':   'top-right'   },
        { 'Bottom Left': 'bottom-left' },
        { 'Bottom Right':'bottom-right'},
        { 'Centered':    'centered'    },
        { 'Stretched':   'stretched'   }
    ],
    default: 'centered',
    title: 'Position'
});

ShowOverlay.attributes.add('enableRotation', { type: 'boolean', default: false, title: 'Activer la rotation' });
ShowOverlay.attributes.add('rotationSpeed',  { type: 'number',  default: 60,   title: 'Vitesse de rotation (deg/s)' });

ShowOverlay.attributes.add('displayDuration', { type: 'number', default: 5,  title: 'Durée d\'affichage (s) – 0 = illimité' });

ShowOverlay.attributes.add('enableFade', { type: 'boolean', default: false, title: 'Fade in / out (250 ms)' });

/* -------------------------------- PRIVATE -------------------------------- */

ShowOverlay.prototype.initialize = function () {
    /* Composant Screen pour dessiner en espace‑écran */
    this.entity.addComponent('screen', {
        screenSpace: true,
        /* On prend la résolution courante de l’app pour éviter toute distorsion
           (utile si tu changes souvent d’appareil/ratio pendant la démo).        */
      referenceResolution: new pc.Vec2(this.app.graphicsDevice.width,
                                  this.app.graphicsDevice.height)

    });

    /* Entité fille qui porte l’image */
    this.imageEntity = new pc.Entity('OverlayImage');
    this.imageEntity.addComponent('element', {
        type: 'image',
        textureAsset: this.imageAsset,
        opacity: 1            // la transparence dépendra du PNG + du fade
    });
    this.entity.addChild(this.imageEntity);




    /* Dès le départ, l’overlay est masqué */
    this.entity.enabled = false;

    /* Gestion des événements externes */
    this.app.on(this.showEvent, this._onShow, this);
    if (this.stopEvent.trim() !== '') {
        this.app.on(this.stopEvent, this._onHide, this);
        this._autoHide = false;
    } else {
        this._autoHide = (this.displayDuration > 0);
    }

    this._timer = 0;
    this._rotationAngle = 0;
};

/* ------------------------------------------------------------------------- */

ShowOverlay.prototype.update = function (dt) {
    if (this.entity.enabled) {
        /* Timer d’auto‑fermeture */
        if (this._autoHide) {
            this._timer -= dt;
            if (this._timer <= 0) this._hide();
        }

        /* Rotation */
        if (this.enableRotation) {
            this._rotationAngle += this.rotationSpeed * dt;
            this.imageEntity.setLocalEulerAngles(0, 0, this._rotationAngle);
        }
    }
};

/* =============================== EVENTS ================================== */

ShowOverlay.prototype._onShow = function () {
    this._applyPosition(this.position);



    /* Affiche (avec fade éventuel) */
/*
    this.entity.enabled = true;
    if (this.enableFade) {
        this.imageEntity.element.opacity = 0;
        this.imageEntity.tween(this.imageEntity.element)
            .to({ opacity: 1 }, 0.25, pc.Linear)
            .start();
    } else {
        this.imageEntity.element.opacity = 1;
    }
*/

// FADE‑IN (250 ms)
this.imageEntity.element.opacity = 0;
this.imageEntity
    .tween(this.imageEntity.element)
    .to({ opacity: 1 }, 0.550, pc.Linear)
    .start();                                 // rien à chaîner après, donc OK



    /* Démarre (ou non) le compte à rebours */
    if (this.displayDuration > 0 && this._autoHide)
        this._timer = this.displayDuration;
};

ShowOverlay.prototype._onHide = function () {
    this._hide();
};

/* ---------------------------- HELPER : HIDE ------------------------------ */

ShowOverlay.prototype._hide = function () {
    if (!this.entity.enabled) return;
/*
    if (this.enableFade) {
        this.imageEntity.tween(this.imageEntity.element)
            .to({ opacity: 0 }, 0.25, pc.Linear)
            .on('complete', function () { this.entity.enabled = false; }.bind(this))
            .start();
    } else {
        this.entity.enabled = false;
    }
*/
// FADE‑OUT (250 ms)
this.imageEntity
    .tween(this.imageEntity.element)
    .to({ opacity: 0 }, 0.505, pc.Linear)
    .onComplete(function () {
        this.entity.enabled = false;
    }.bind(this))
    .start();


};

/* ---------------------- HELPER : POSITIONNEMENT -------------------------- */

ShowOverlay.prototype._applyPosition = function (mode) {

    /* Récupère la taille « native » du PNG (fallback 300×300 si non chargée). */
    var tex = this.imageAsset && this.imageAsset.resource;
    var w   = tex ? tex.width  : 300;
    var h   = tex ? tex.height : 300;

    var el  = this.imageEntity.element;

    switch (mode) {
        /* ---------- Coins ---------- */
        case 'top-left':
            el.anchor = [0, 1, 0, 1];
            el.pivot  = [0, 1];
            break;

        case 'top-right':
            el.anchor = [1, 1, 1, 1];
            el.pivot  = [1, 1];
            break;

        case 'bottom-left':
            el.anchor = [0, 0, 0, 0];
            el.pivot  = [0, 0];
            break;

        case 'bottom-right':
            el.anchor = [1, 0, 1, 0];
            el.pivot  = [1, 0];
            break;

        /* ------------ Centre ----------- */
        case 'centered':
            el.anchor = [0.5, 0.5, 0.5, 0.5];
            el.pivot  = [0.5, 0.5];
            break;

        /* ----------- Plein écran -------- */
        case 'stretched':
            el.anchor = [0, 0, 1, 1];
            el.pivot  = [0.5, 0.5];
            /* Largeur / hauteur = 0 → l’element occupe 100 % de la zone délimitée par l’ancre */
            el.width  = 0;
            el.height = 0;
            return;              // rien d’autre à faire
    }

    /* Pour tous les modes sauf « stretched », on fixe la taille voulue */
    el.width  = w;
    el.height = h;
    this.imageEntity.setLocalPosition(0, 0, 0);   // inutile mais garde la scène propre
};
