// --------------- POST EFFECT DEFINITION --------------- //
/**
 * @class
 * @name TransitionEffect
 * @classdesc Implements a screen-flooding "light" transition effect with irregular edges.
 * @description Creates new instance of the transition post effect.
 * @augments pc.PostEffect
 * @param {pc.GraphicsDevice} graphicsDevice - The graphics device of the application.
 * @property {pc.Color} color The color of the flooding light.
 * @property {number} progress How much of the screen is flooded (0 = rien, 1 = tout l’écran).
 * @property {number} noiseScale Facteur d’échelle pour le bruit.
 * @property {number} speed Vitesse d’évolution du bruit dans le temps (animation).
 */
function TransitionEffect(graphicsDevice) {
    pc.PostEffect.call(this, graphicsDevice);

    // Cet effet ne dépend pas du depthBuffer
    this.needsDepthBuffer = false;

    // Paramètres
    this.color = new pc.Color(1, 1, 1, 1);  // Couleur par défaut (blanc)
    this.progress = 0.0;
    this.noiseScale = 3.0;
    this.speed = 0.2;
    this.time = 0.0;

    // -- FRAGMENT SHADER --
    // On utilise un bruit très simple (hash-based) et on mixe la scène avec la couleur de transition.
    var fshader = [
        "varying vec2 vUv0;",
        "uniform sampler2D uColorBuffer;",    // scène
        "uniform vec4 uColor;",              // couleur de l'effet
        "uniform float uProgress;",          // progression de la transition [0..1]
        "uniform float uNoiseScale;",        // échelle du bruit
        "uniform float uTime;",             // temps écoulé (pour animer le bruit)
        "uniform float uSpeed;",            // vitesse d'évolution du bruit
        "",
        // Fonction de bruit pseudo-aléatoire (hash)
        "float rand(vec2 co){",
        "    return fract(sin(dot(co.xy ,vec2(12.9898,78.233))) * 43758.5453);",
        "}",
        "",
        "void main() {",
        "    // Récupération de la couleur de la scène",
        "    vec4 sceneColor = texture2D(uColorBuffer, vUv0);",
        "",
        "    // On calcule la distance radiale par rapport au centre de l’écran",
        "    vec2 center = vec2(0.5, 0.5);",
        "    float dist = distance(vUv0, center);",
        "",
        "    // On calcule un bruit pour perturber les bords",
        "    // Pour l'animer: on ajoute (uTime*uSpeed) à la coordonnée de bruit.",
        "    float noiseVal = rand(vUv0 * uNoiseScale + uTime * uSpeed);",
        "",
        "    // Idée : plus uProgress est grand, plus l’écran est « noyé » dans la lumière.",
        "    // On va thresholder sur (dist + un peu de bruit) par rapport à uProgress.",
        "",
        "    // Par exemple, on fait en sorte que quand dist + noiseVal < (1.0 - uProgress), on noie cette zone.",
        "    // Ajustez la formule à votre goût pour contrôler la forme des bords.",
        "    float cutoff = 1.0 - uProgress;  // se réduit au fur et à mesure que progress augmente",
        "    float floodFactor = smoothstep(cutoff, cutoff - 0.2, dist + 0.4*noiseVal);",
        "",
        "    // floodFactor sera proche de 0 dans les zones noyées par la lumière, et proche de 1 où la scène reste visible",
        "    // On veut au contraire un alpha fort là où c’est « inondé »",
        "    float alpha = 1.0 - floodFactor;",
        "",
        "    // Couleur finale : mix entre la scène (pondérée par 1-alpha) et la couleur de l’effet (pondérée par alpha).",
        "    vec3 finalColor = mix(sceneColor.rgb, uColor.rgb, alpha);",
        "    gl_FragColor = vec4(finalColor, 1.0);",
        "}"
    ].join("\n");

    // Création du shader
    this.shader = pc.createShaderFromCode(
        graphicsDevice,
        pc.PostEffect.quadVertexShader, 
        fshader,
        'TransitionShader',
        { aPosition: pc.SEMANTIC_POSITION }
    );
}

TransitionEffect.prototype = Object.create(pc.PostEffect.prototype);
TransitionEffect.prototype.constructor = TransitionEffect;

Object.assign(TransitionEffect.prototype, {
    render: function (inputTarget, outputTarget, rect) {
        var device = this.device;
        var scope = device.scope;

        // Mise à jour du temps (pour animer le bruit) :
        this.time += this.speed * (1.0 / 60.0); // hypothèse ~ 60fps

        // On envoie les uniformes
        scope.resolve("uColorBuffer").setValue(inputTarget.colorBuffer);
        scope.resolve("uColor").setValue(this.color.data);
        scope.resolve("uProgress").setValue(this.progress);
        scope.resolve("uNoiseScale").setValue(this.noiseScale);
        scope.resolve("uTime").setValue(this.time);
        scope.resolve("uSpeed").setValue(this.speed);

        // Dessin
        this.drawQuad(outputTarget, this.shader, rect);
    }
});
