// --------------- POST EFFECT DEFINITION --------------- //
// On suppose que BangEffect est déjà défini quelque part (non reproduit ici).
// Il expose au moins les propriétés: time, active, etc.

// ─────────────────────────────────────────────────────────────────────────────
// SCRIPT Bang
// ─────────────────────────────────────────────────────────────────────────────
var Bang = pc.createScript('bang');

/**
 * ATTRIBUTS
 */
// Nom de l’événement qui active l’effet (ex: "bang")
Bang.attributes.add('enableEventName', {
    type: 'string',
    default: 'bang',
    title: 'Enable Event',
    description: 'Nom de l’événement qui activera ce post-effect'
});

// Nom de l’événement qui désactive l’effet (optionnel)
Bang.attributes.add('disableEventName', {
    type: 'string',
    default: '',
    title: 'Disable Event (optional)',
    description: 'Nom de l’événement qui désactivera ce post-effect'
});

// Durée d'exécution en secondes (0 => boucle infinie)
Bang.attributes.add('duration', {
    type: 'number',
    default: 0,
    title: 'Duration (0=loop)'
});

// Démarrage automatique
Bang.attributes.add('autoStart', {
    type: 'boolean',
    default: false,
    title: 'Auto Start'
});


/**
 * INITIALISATION
 */
Bang.prototype.initialize = function() {
    // 1) Crée l'instance du post-effect
    this.effect = new BangEffect(this.app.graphicsDevice);
    // Par défaut, on met effect.active = false
    this.effect.active = false;

    // 2) Référence vers la "queue" de postEffects
    this.queue = this.entity.camera.postEffects;

    // 3) Variable interne pour savoir si l’effet est dans la queue
    this._effectAdded = false;

    // 4) Variables internes pour gérer la durée
    //    On stocke le temps de départ du “bang” (pour comparer ensuite)
    this.bangStartTime = -1;

    // 5) Écoute des événements
    //    => Quand on reçoit "enableEventName", on active l’effet
    if (this.enableEventName) {
        this.app.on(this.enableEventName, this.enableBang, this);
    }
    //    => Quand on reçoit "disableEventName", on le désactive
    if (this.disableEventName && this.disableEventName.length > 0) {
        this.app.on(this.disableEventName, () => {
            this.disableBang(false);  // on ne détruit pas l’effet
        }, this);
    }

    // 6) Activation / désactivation du script
    //    => si le script est désactivé, on retire l’effet
    this.on('state', function(enabled) {
        if (!enabled) {
            this.disableBang(false);
        } else {
            // Si on ré-active le script et qu’on avait déjà l’effet
            if (this._effectAdded) {
                this.enableBang();
            }
        }
    });

    // 7) Nettoyage quand le script est détruit
    this.on('destroy', function() {
        this.disableBang(true);
    });

    // 8) AutoStart ?
    if (this.autoStart) {
        this.enableBang();
    }
};


/**
 * MISE À JOUR (CHAQUE FRAME)
 */
Bang.prototype.update = function(dt) {
    // On ne fait la logique que si l’effet est “actif”
    if (this.effect.active) {
        // Incrémente le temps interne
        this.effect.time += dt;

        // Si on a une durée > 0, on check si on doit stopper
        if (this.duration > 0) {
            let elapsed = this.effect.time - this.bangStartTime;
            if (elapsed >= this.duration) {
                // On coupe l'effet
                this.effect.active = false;
            }
        }
    }
};


/**
 * MÉTHODES PERSONNALISÉES
 */

// ENABLE : ajoute l’effet à la queue, et “triggerBang”
Bang.prototype.enableBang = function() {
    // Si le script est désactivé, on ne fait rien
    if (!this.enabled) return;

    // On ajoute l’effet s’il n’est pas déjà là
    if (!this._effectAdded) {
        this.queue.addEffect(this.effect);
        this._effectAdded = true;
    }

    // On déclenche le “bang”
    this.triggerBang();
};

// DISABLE : retire l’effet de la queue
// param “destroy” => si true, on détruit l’effet
Bang.prototype.disableBang = function(destroy) {
    if (this._effectAdded) {
        this.queue.removeEffect(this.effect);
        this._effectAdded = false;
    }
    if (destroy && this.effect && this.effect._destroy) {
        this.effect._destroy();
    }
    // Et on coupe l’animation
    this.effect.active = false;
};

/**
 * TRIGGERBANG
 * Méthode publique pour relancer l'effet "bang".
 */
Bang.prototype.triggerBang = function() {
    // On réinitialise le compteur de temps
    this.effect.time = 0;
    this.bangStartTime = 0; 
    // On active
    this.effect.active = true;
};
