// --------------- POST EFFECT DEFINITION --------------- //
// On suppose que Bang2Effect est déjà défini quelque part (non reproduit ici).
// Il expose au moins les propriétés: time, factor, etc.

/**
 * Script Bang2
 * - Déclenche le post-effect Bang2 via un événement "enableEventName"
 * - Peut également stopper l'effet via un événement "disableEventName" (optionnel)
 * - Gère un fade (0 -> 1) sur "factor".
 */
var Bang2 = pc.createScript('bang2');

/**
 * ATTRIBUTS
 */
// Événement qui active le post-effect
Bang2.attributes.add('enableEventName', {
    type: 'string',
    default: 'bang2',
    title: 'Enable Event',
    description: 'Nom de l’événement qui activera ce post-effect et lancera le fade.'
});

// Événement (optionnel) qui désactive le post-effect
Bang2.attributes.add('disableEventName', {
    type: 'string',
    default: '',
    title: 'Disable Event (optional)',
    description: 'Nom de l’événement qui désactivera ce post-effect'
});

// Vitesse d’animation du paramètre "time"
Bang2.attributes.add('speed', {
    type: 'number',
    default: 1,
    title: 'Time Speed'
});

// Durée du fade (secondes) pour aller de factor=0 à factor=1
Bang2.attributes.add('fadeDuration', {
    type: 'number',
    default: 2,
    title: 'Fade Duration (seconds)'
});


/**
 * INITIALISATION
 */
Bang2.prototype.initialize = function() {
    // 1) Crée l’effet
    this.effect = new Bang2Effect(this.app.graphicsDevice);

    // 2) Récupère la file de postEffects de la caméra
    this.queue = this.entity.camera.postEffects;

    // 3) État interne pour le fade
    this.fading = false;
    this.fadeTime = 0;    // on incrémentera 0..fadeDuration
    this._bang2Added = false; // Pour savoir si l’effet est actif dans la queue

    // 4) Écoute l’événement "enableEventName"
    if (this.enableEventName) {
        this.app.on(this.enableEventName, this.enableBang2, this);
    }
    // Écoute l’événement "disableEventName" (si spécifié)
    if (this.disableEventName && this.disableEventName.length > 0) {
        this.app.on(this.disableEventName, () => {
            // On le désactive sans détruire, pour pouvoir le relancer plus tard
            this.disableBang2(false);
        }, this);
    }

    // 5) Gestion de l’activation/désactivation du script
    this.on('state', function(enabled){
        if (!enabled) {
            // Retirer l’effet s’il était actif
            this.disableBang2(false);
        } else {
            // S’il avait été déclenché auparavant, on le ré-active
            if (this._bang2Added) {
                this.enableBang2();
            }
        }
    });

    // 6) Nettoyage quand le script est détruit
    this.on('destroy', function(){
        // On le désactive en détruisant l’effet
        this.disableBang2(true);
    });
};


/**
 * APPELÉ À CHAQUE FRAME
 */
Bang2.prototype.update = function(dt) {
    // Incrémente "time"
    this.effect.time += dt * this.speed;

    // Gestion du fade si en cours
    if (this.fading) {
        this.fadeTime += dt;
        let f = this.fadeTime / this.fadeDuration;
        if (f >= 1) {
            f = 1;
            this.fading = false;
        }
        this.effect.factor = f;
    }
};


/**
 * MÉTHODES PERSONNALISÉES
 */

// "enableBang2" => Active l’effet si le script est enabled, puis lance le fade
Bang2.prototype.enableBang2 = function() {
    // S’il est désactivé, on ne fait rien
    if (!this.enabled) return;

    // Ajouter l’effet si pas déjà présent
    if (!this._bang2Added) {
        this.queue.addEffect(this.effect);
        this._bang2Added = true;
    }

    // Lance (ou relance) le fade
    this.fading = true;
    this.fadeTime = 0;
    this.effect.factor = 0;  // on force l’effet à partir de 0
};

// "disableBang2" => Retire l’effet
// Paramètre 'destroy' = true => on détruit l’effet (à l’arrêt complet du script)
Bang2.prototype.disableBang2 = function(destroy) {
    if (this._bang2Added) {
        this.queue.removeEffect(this.effect);
        this._bang2Added = false;
    }
    if (destroy && this.effect._destroy) {
        this.effect._destroy();
    }
};
