var MovementOnEvent = pc.createScript('movementOnEvent');

/**
 * ─────────────────────────────────────────────────────────────────────────────
 *  ATTRIBUTS CONFIGURABLES DANS L'ÉDITEUR
 * ─────────────────────────────────────────────────────────────────────────────
 */

// 1) Nom de l'événement PlayCanvas qui déclenche le déplacement
MovementOnEvent.attributes.add('triggerEventName', {
    type: 'string',
    default: 'moveEntity',
    title: 'Événement déclencheur',
    description: 'Nom de l’événement qui va déclencher le déplacement'
});

// 2) Nom de l'événement à émettre quand le déplacement est terminé (optionnel)
MovementOnEvent.attributes.add('onCompleteEventName', {
    type: 'string',
    default: '',
    title: 'Événement de fin',
    description: 'Si non vide, on émettra cet événement quand le mouvement est fini'
});

// 2-bis) Délai en millisecondes avant d’émettre l’événement de fin
MovementOnEvent.attributes.add('onCompleteDelayMs', {
    type: 'number',
    default: 0,
    title: 'Délai fin (ms)',
    description: 'Délai en millisecondes avant de déclencher l’événement de fin'
});

// 3) Vitesse de déplacement (unités/secondes)
MovementOnEvent.attributes.add('speed', {
    type: 'number',
    default: 2,
    title: 'Vitesse (unités/sec)'
});

// 4) Délai (en millisecondes) avant de commencer réellement le mouvement
MovementOnEvent.attributes.add('startDelayMs', {
    type: 'number',
    default: 0,
    title: 'Délai (ms)'
});

// 5) Position finale par défaut
MovementOnEvent.attributes.add('endPosition', {
    type: 'vec3',
    default: [2.1814935, 58.5488281, 12.1005067],
    title: 'Position finale'
});

// 6) Direction finale (une fois arrivé). 
MovementOnEvent.attributes.add('endDirection', {
    type: 'vec3',
    default: [-0.0240331, 0.9841956, -0.1754463],
    title: 'Direction finale'
});

// 7) Entité à « regarder » pendant le déplacement (optionnel)
MovementOnEvent.attributes.add('target', {
    type: 'entity',
    title: 'Cible à regarder',
    description: 'Si spécifié, l’entité regardera cette cible pendant le trajet.'
});

// 8) Type de lissage (easing) pour le mouvement
MovementOnEvent.attributes.add('easingType', {
    type: 'string',
    default: 'Linear',
    title: 'Easing',
    description: 'Type de lissage / courbe d’animation',
    enum: [
        { 'Linear'       : 'Linear' },
        { 'EaseInQuad'   : 'EaseInQuad' },
        { 'EaseOutQuad'  : 'EaseOutQuad' },
        { 'EaseInOutQuad': 'EaseInOutQuad' },
        { 'ElasticOut'   : 'ElasticOut' },
        { 'BounceOut'    : 'BounceOut' }
    ]
});


/**
 * ─────────────────────────────────────────────────────────────────────────────
 *  INITIALISATION
 * ─────────────────────────────────────────────────────────────────────────────
 */
MovementOnEvent.prototype.initialize = function() {
    // Variables pour gérer le délai avant le mouvement
    this.waiting = false;
    this.waitTime = 0; // compteur d’attente (ms)

    // Variables pour gérer le déplacement
    this.moving = false;
    this.travelTime = 0; // durée totale du déplacement (s)
    this.elapsed = 0;    // temps écoulé depuis le début (s)

    // Positions de départ/arrivée
    this.startPos = new pc.Vec3();
    this.endPos   = new pc.Vec3();

    // Dictionnaire de fonctions d'easing
    this.easingFunctions = {
        Linear: function(t) {
            return t;
        },
        EaseInQuad: function(t) {
            return t * t;
        },
        EaseOutQuad: function(t) {
            return t * (2 - t);
        },
        EaseInOutQuad: function(t) {
            return (t < 0.5) ? (2 * t * t) : (1 - Math.pow(-2 * t + 2, 2) / 2);
        },
        ElasticOut: function(t) {
            var c4 = (2 * Math.PI) / 3;
            return (t === 0) ? 0 :
                   (t === 1) ? 1 :
                   Math.pow(2, -10 * t) * Math.sin((t * 10 - 0.75) * c4) + 1;
        },
        BounceOut: function(t) {
            var n1 = 7.5625;
            var d1 = 2.75;
            if (t < 1 / d1) {
                return n1 * t * t;
            } else if (t < 2 / d1) {
                t -= 1.5 / d1;
                return n1 * t * t + 0.75;
            } else if (t < 2.5 / d1) {
                t -= 2.25 / d1;
                return n1 * t * t + 0.9375;
            } else {
                t -= 2.625 / d1;
                return n1 * t * t + 0.984375;
            }
        }
    };

    // Variables pour gérer le délai d’envoi de l’événement de fin
    this.endWaiting = false;
    this.endWaitTimer = 0;

    // Écoute l'événement "triggerEventName" pour lancer le déplacement
    if (this.triggerEventName && this.triggerEventName.length > 0) {
        this.app.on(this.triggerEventName, this.startMovement, this);
    }
};


/**
 * ─────────────────────────────────────────────────────────────────────────────
 *  MÉTHODE DE DÉMARRAGE DU DÉPLACEMENT
 * ─────────────────────────────────────────────────────────────────────────────
 */
MovementOnEvent.prototype.startMovement = function() {
    // On définit la position de départ comme la position actuelle
    this.startPos.copy(this.entity.getPosition());

    // On définit la position finale depuis l’attribut
    this.endPos.copy(this.endPosition);

    // Calcul de la distance
    var distance = this.startPos.distance(this.endPos);

    // Calcul du temps de déplacement
    this.travelTime = (this.speed > 0) ? (distance / this.speed) : 0;
    this.elapsed = 0;

    // Gérer le délai initial
    if (this.startDelayMs > 0) {
        this.waiting = true;
        this.waitTime = 0;
        this.moving = false;
    } else {
        this.waiting = false;
        this.moving = true;
    }

    // Au cas où on relance un mouvement, on remet
    // le délai de fin à zéro
    this.endWaiting = false;
    this.endWaitTimer = 0;
};


/**
 * ─────────────────────────────────────────────────────────────────────────────
 *  UPDATE (CHAQUE FRAME)
 * ─────────────────────────────────────────────────────────────────────────────
 */
MovementOnEvent.prototype.update = function(dt) {
    // 1) Gérer le délai avant le mouvement
    if (this.waiting) {
        this.waitTime += dt * 1000;
        if (this.waitTime >= this.startDelayMs) {
            // Fin du délai, on démarre
            this.waiting = false;
            this.moving = true;
            this.elapsed = 0;
        }
    }

    // 2) Gérer le déplacement
    if (this.moving) {
        this.elapsed += dt;

        // Ratio linéaire [0..1]
        var rawRatio = (this.travelTime > 0) ? (this.elapsed / this.travelTime) : 1;
        if (rawRatio > 1) rawRatio = 1;

        // Easing
        var easeFn = this.easingFunctions[this.easingType] || this.easingFunctions.Linear;
        var easedRatio = easeFn(rawRatio);

        // Interpolation de la position
        var newPos = new pc.Vec3().lerp(this.startPos, this.endPos, easedRatio);
        this.entity.setPosition(newPos);

        // Orientation vers la target si besoin
        if (this.target && rawRatio < 1) {
            this.entity.lookAt(this.target.getPosition());
        }

        // Fin du mouvement ?
        if (rawRatio === 1) {
            this.moving = false;

            // Fixe la position pile
            this.entity.setPosition(this.endPos);

            // Orientation finale
            if (this.endDirection) {
                var finalLookAt = this.endPos.clone().add(this.endDirection);
                this.entity.lookAt(finalLookAt);
            }

            // Si un événement de fin est spécifié, on prépare un éventuel délai
            if (this.onCompleteEventName && this.onCompleteEventName.length > 0) {
                if (this.onCompleteDelayMs > 0) {
                    this.endWaiting = true;
                    this.endWaitTimer = 0;
                } else {
                    // Pas de délai => on déclenche immédiatement
                    this.app.fire(this.onCompleteEventName);
                }
            }
        }
    }

    // 3) Gérer le délai avant l’événement de fin (si endWaiting)
    if (this.endWaiting) {
        this.endWaitTimer += dt * 1000;
        if (this.endWaitTimer >= this.onCompleteDelayMs) {
            // On déclenche l’événement
            this.endWaiting = false;
            this.app.fire(this.onCompleteEventName);
        }
    }
};
