var TimelineActivator = pc.createScript('timelineActivator');

// Événement de démarrage (par défaut : "startTimeline")
TimelineActivator.attributes.add('startEvent', {
    type: 'string',
    default: 'startTimeline',
    title: 'Événement de démarrage'
});

// Tableau d'items (événements programmés)
TimelineActivator.attributes.add('items', {
    type: 'json',
    title: 'Timeline Items',
    array: true,
    schema: [
        {
            name: 'time',
            type: 'number',
            default: 0,
            title: 'Temps (s)'
        },
        {
            name: 'entity',
            type: 'entity',
            title: 'Entité à activer'
        },
        {
            name: 'duration',
            type: 'number',
            default: 0,
            title: 'Durée active (s)'
        },
        {
            name: 'eventName',
            type: 'string',
            default: '',
            title: 'Nom de l\'événement (optionnel)'
        }
    ]
});

TimelineActivator.prototype.initialize = function() {
    // Indicateur pour savoir si la timeline est active
    this.isActive = false;
    
    // Temps écoulé depuis le démarrage de la timeline
    this.timeElapsed = 0;

    // Copie du tableau d'items, puis tri par temps croissant
    // (Afin de s'assurer que les activations se font dans l'ordre prévu)
    this.timeline = this.items.slice().sort(function(a, b) {
        return a.time - b.time;
    });
    
    // Pour chaque item, on ajoute un flag "triggered" à false
    // afin de savoir s’il a déjà été déclenché.
    for (var i = 0; i < this.timeline.length; i++) {
        this.timeline[i].triggered = false;
    }

    // Écoute l’événement de démarrage
    this.app.on(this.startEvent, this.startTimeline, this);
};

// Lance la timeline
TimelineActivator.prototype.startTimeline = function() {
    if (!this.isActive) {
        this.isActive = true;
        this.timeElapsed = 0;
        console.log("Timeline activée suite à l'événement: " + this.startEvent);
    }
};

TimelineActivator.prototype.update = function(dt) {
    // Si la timeline n'est pas active, on ne fait rien
    if (!this.isActive) return;

    // Incrémente le temps écoulé
    this.timeElapsed += dt;

    // Parcours de la liste des items
    for (var i = 0; i < this.timeline.length; i++) {
        var item = this.timeline[i];

        // 1) Vérifie si on doit activer l'entité (et éventuellement émettre l'événement)
        if (!item.triggered && this.timeElapsed >= item.time) {
            item.triggered = true;

            // Active l'entité si elle existe
            if (item.entity) {
                item.entity.enabled = true;
            }

            // Émet un événement PlayCanvas si "eventName" est défini
            if (item.eventName && item.eventName.trim() !== "") {
                this.app.fire(item.eventName);
            }

            // Si on a une durée > 0, on enregistre l’instant où on doit la détruire
            if (item.duration > 0) {
                item.destroyTime = item.time + item.duration;
            }
        }

        // 2) Vérifie si on doit détruire l'entité après la durée écoulée
        if (item.triggered && item.duration > 0 && this.timeElapsed >= item.destroyTime) {
            if (item.entity && !item.entity.destroyed) {
                item.entity.destroy();
            }
            // On peut retirer l'item de la timeline pour éviter de refaire le check
            this.timeline.splice(i, 1);
            i--;
        }
    }
};
