/**
 * Script: autoBloom
 * 
 * Fonctionnement:
 *   - Attaché à CHAQUE caméra.
 *   - Crée et active automatiquement l'effet Bloom quand la caméra devient active.
 *   - Désactive automatiquement l'effet quand la caméra devient inactive.
 *   - Pré-charge les ressources du Bloom pour éviter les blocages lors du changement de caméra.
 *
 * ⚠️ Nécessite obligatoirement un composant Camera sur l'entité.
 */

// Définition du script
var AutoBloom = pc.createScript('autoBloom');

// Liste statique de toutes les instances autoBloom
AutoBloom._allInstances = [];

// Paramètres de l'effet
AutoBloom.attributes.add('bloomIntensity', { type: 'number', default: 1.25, min: 0, title: 'Intensity' });
AutoBloom.attributes.add('bloomThreshold', { type: 'number', default: 0.25, min: 0, max: 1, title: 'Threshold' });
AutoBloom.attributes.add('blurAmount', { type: 'number', default: 4, min: 1, title: 'Blur amount' });

// INITIALISATION
AutoBloom.prototype.initialize = function() {
    console.log("[autoBloom] initialize() sur caméra:", this.entity.name);

    AutoBloom._allInstances.push(this);

    this.effect = new BloomEffect(this.app.graphicsDevice);
    this.effect.bloomThreshold = this.bloomThreshold;
    this.effect.blurAmount     = this.blurAmount;
    this.effect.bloomIntensity = this.bloomIntensity;

    this.cameraComponent = this.entity.camera;
    if (!this.cameraComponent) {
        console.error("[autoBloom] Aucun composant caméra trouvé sur l'entité", this.entity.name);
        return;
    }

    this._effectAdded = false;

    this.on('enable', this.onEnable, this);
    this.on('disable', this.onDisable, this);

    if (this.entity.enabled) {
        this.onEnable();
    }

    // ⚠️ Pré-chauffage immédiat des ressources Bloom
    this.prewarmBloom();
};

// MÉTHODE DE PRÉ-CHAUFFAGE DU BLOOM
AutoBloom.prototype.prewarmBloom = function () {
    if (!this.cameraComponent) return;

    console.log("[autoBloom] Pré-chauffage BloomEffect pour la caméra:", this.entity.name);

    var device = this.app.graphicsDevice;
    var width = device.width;
    var height = device.height;

    // Texture temporaire pour préchauffage
    var tempColorBuffer = new pc.Texture(device, {
        width: width,
        height: height,
        format: pc.PIXELFORMAT_RGBA8,
        mipmaps: false,
        minFilter: pc.FILTER_LINEAR,
        magFilter: pc.FILTER_LINEAR,
        addressU: pc.ADDRESS_CLAMP_TO_EDGE,
        addressV: pc.ADDRESS_CLAMP_TO_EDGE
    });

    // RenderTarget temporaire
    var tempTarget = new pc.RenderTarget({
        colorBuffer: tempColorBuffer,
        depth: true
    });

    // Force un rendu fictif pour charger immédiatement les ressources GPU
    this.effect.render(tempTarget, tempTarget, new pc.Vec4(0, 0, width, height));

    // Nettoyage des ressources temporaires
    tempTarget.destroy();
    tempColorBuffer.destroy();

    console.log("[autoBloom] Pré-chauffage terminé pour caméra:", this.entity.name);
};

// DESTRUCTION DU SCRIPT
AutoBloom.prototype.destroy = function() {
    console.log("[autoBloom] destroy() sur caméra:", this.entity.name);

    var idx = AutoBloom._allInstances.indexOf(this);
    if (idx !== -1) {
        AutoBloom._allInstances.splice(idx, 1);
    }
    this.removeBloom();
};

// AJOUT DE L'EFFET BLOOM
AutoBloom.prototype.addBloom = function() {
    if (!this._effectAdded && this.cameraComponent) {
        console.log("[autoBloom] Ajout Bloom à la caméra:", this.entity.name);
        this.cameraComponent.postEffects.addEffect(this.effect);
        this._effectAdded = true;
    }
};

// SUPPRESSION DE L'EFFET BLOOM
AutoBloom.prototype.removeBloom = function() {
    if (this._effectAdded && this.cameraComponent) {
        console.log("[autoBloom] Retrait Bloom de la caméra:", this.entity.name);
        this.cameraComponent.postEffects.removeEffect(this.effect);
        this._effectAdded = false;
    }
};

// MÉTHODE STATIQUE POUR DÉSACTIVER LE BLOOM SUR LES AUTRES CAMÉRAS
AutoBloom.disableAllBut = function(instanceConserve) {
    console.log("[autoBloom] Désactivation Bloom partout sauf:", instanceConserve.entity.name);

    for (var i = 0; i < AutoBloom._allInstances.length; i++) {
        var inst = AutoBloom._allInstances[i];
        if (inst !== instanceConserve) {
            inst.removeBloom();
        }
    }
};

// ACTIVATION DE LA CAMÉRA
AutoBloom.prototype.onEnable = function() {
    console.log("[autoBloom] onEnable() activé sur caméra:", this.entity.name);

    AutoBloom.disableAllBut(this);
    this.addBloom();
};

// DÉSACTIVATION DE LA CAMÉRA

AutoBloom.prototype.onDisable = function() {
    console.log("[autoBloom] onDisable() activé sur caméra:", this.entity.name);

    this.removeBloom();

    // ⚠️ Réinitialise l'état de l'effet pour permettre sa réactivation plus tard
    this._effectAdded = false;
};