
var SegmentedVibration = pc.createScript('segmentedVibration');

/**
 * ────────────────────────────────────────────────────────────────────────────────
 * 1) ATTRIBUTS EXPOSÉS (UI PlayCanvas)
 * ────────────────────────────────────────────────────────────────────────────────
 */

// Amplitude (distance) min/max
SegmentedVibration.attributes.add('amplitudeMin', {
    type: 'number',
    default: 0.01,
    title: 'Amplitude Min'
});
SegmentedVibration.attributes.add('amplitudeMax', {
    type: 'number',
    default: 0.1,
    title: 'Amplitude Max'
});

// Vitesse (fréquence) min/max
SegmentedVibration.attributes.add('speedMin', {
    type: 'number',
    default: 1,
    title: 'Vitesse Min'
});
SegmentedVibration.attributes.add('speedMax', {
    type: 'number',
    default: 5,
    title: 'Vitesse Max'
});

// Offset (décalage de phase) min/max
SegmentedVibration.attributes.add('offsetMin', {
    type: 'number',
    default: 0,
    title: 'Offset Min'
});
SegmentedVibration.attributes.add('offsetMax', {
    type: 'number',
    default: 3,
    title: 'Offset Max'
});

// Toutes les X secondes, on re-pioche des valeurs aléatoires
SegmentedVibration.attributes.add('randomInterval', {
    type: 'number',
    default: 2,
    title: 'Interval Re-pick (s)'
});

/**
 * ────────────────────────────────────────────────────────────────────────────────
 * 2) INITIALISATION : SHADER + MATÉRIAU
 * ────────────────────────────────────────────────────────────────────────────────
 */
SegmentedVibration.prototype.initialize = function() {
    this.time = 0;
    this.timer = 0;

    // Prépare les premières valeurs random
    this._randomizeParams();

    // Construction du shader (vertex + fragment), avec BACKTICKS pour plus de lisibilité
    const gd = this.app.graphicsDevice;

    const vertexShader = `
        attribute vec3 aPosition;
        attribute vec2 aUv0;

        uniform mat4 matrix_model;
        uniform mat4 matrix_viewProjection;

        uniform float uTime;
        uniform float uAmplitude;
        uniform float uSpeed;
        uniform float uOffset;

        varying vec2 vUv0;

        void main(void) {
            vec3 pos = aPosition;

            // Ex. : vibration sin/cos en fonction de (pos.x, pos.y, temps, offset)
            pos.x += sin(pos.y * uSpeed + uOffset + uTime * uSpeed) * uAmplitude;
            pos.y += cos(pos.x * uSpeed + uOffset + uTime * uSpeed) * uAmplitude;

            // Transformation world + viewProjection
            vec4 worldPos = matrix_model * vec4(pos, 1.0);
            gl_Position = matrix_viewProjection * worldPos;

            vUv0 = aUv0;
        }
    `;

    const fragmentShader = `
        precision ${gd.precision} float;
        varying vec2 vUv0;

        void main(void) {
            // Simple coloration par UV
            gl_FragColor = vec4(vUv0, 0.0, 1.0);
        }
    `;

    const shaderDefinition = {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION,
            aUv0: pc.SEMANTIC_TEXCOORD0
        },
        uniforms: [
            { name: "matrix_model",          semantic: pc.SEMANTIC_MODEL,          type: pc.UNIFORMTYPE_MAT4 },
            { name: "matrix_viewProjection", semantic: pc.SEMANTIC_VIEWPROJECTION, type: pc.UNIFORMTYPE_MAT4 },
            { name: "uTime",       type: pc.UNIFORMTYPE_FLOAT },
            { name: "uAmplitude",  type: pc.UNIFORMTYPE_FLOAT },
            { name: "uSpeed",      type: pc.UNIFORMTYPE_FLOAT },
            { name: "uOffset",     type: pc.UNIFORMTYPE_FLOAT }
        ],
        vshader: vertexShader,
        fshader: fragmentShader
    };

    this.shader = new pc.Shader(gd, shaderDefinition);

    this.material = new pc.Material();
    this.material.shader = this.shader;
    this.material.update();

    // Appliquer le matériau
    const renderOrModel = this.entity.render || this.entity.model;
    if (renderOrModel && renderOrModel.meshInstances) {
        for (let i = 0; i < renderOrModel.meshInstances.length; i++) {
            renderOrModel.meshInstances[i].material = this.material;
        }
    } else {
        console.warn("[SegmentedVibration] Aucune meshInstance trouvée sur l'entité.");
    }
};

/**
 * ────────────────────────────────────────────────────────────────────────────────
 * 3) UPDATE : GESTION DU TEMPS, RE-PICK ALÉATOIRE, ET ENVOI UNIFORMS
 * ────────────────────────────────────────────────────────────────────────────────
 */
SegmentedVibration.prototype.update = function(dt) {
    this.time += dt;
    this.timer += dt;

    // Si on dépasse randomInterval, on rechoisit de nouvelles valeurs
    if (this.timer >= this.randomInterval) {
        this.timer = 0;
        this._randomizeParams();
    }

    // Envoie les params au shader
    this.material.setParameter("uTime", this.time);
    this.material.setParameter("uAmplitude", this.currentAmplitude);
    this.material.setParameter("uSpeed", this.currentSpeed);
    this.material.setParameter("uOffset", this.currentOffset);
};

/**
 * ────────────────────────────────────────────────────────────────────────────────
 * 4) FONCTION PRIVÉE : CHOISIT DE NOUVEAUX PARAMS ALÉATOIRES
 * ────────────────────────────────────────────────────────────────────────────────
 */
SegmentedVibration.prototype._randomizeParams = function() {
    this.currentAmplitude = pc.math.random(this.amplitudeMin, this.amplitudeMax);
    this.currentSpeed     = pc.math.random(this.speedMin,     this.speedMax);
    this.currentOffset    = pc.math.random(this.offsetMin,    this.offsetMax);
};
