function StarfieldDepthEffect(graphicsDevice) {
    pc.PostEffect.call(this, graphicsDevice);

    // On a besoin de la depth buffer pour déterminer les zones "vides"
    this.needsDepthBuffer = true;

    // Variables pour l’animation et le fade
    this.time = 0;
    this.fade = 0;
    // Nouvelle variable pour le décalage de la caméra (en UV)
    this.camOffset = new pc.Vec2(0, 0);

    const fshader = `  
precision ${graphicsDevice.precision} float;
${pc.shaderChunks.screenDepthPS}

varying vec2 vUv0;

uniform sampler2D uColorBuffer;
uniform float uTime;
uniform float uFade;
uniform vec2  uResolution;
uniform vec2  uCamOffset;   // Déplacement lié à la caméra

// ---------------------------------------------------------
// Définition des constantes
// ---------------------------------------------------------
#define NS 800.0               // Nombre d'étoiles générées dans le ciel
#define HASHSCALE1 0.1031      // Constante utilisée pour le bruit dans les nuages

// ---------------------------------------------------------
// Fonctions dédiées aux étoiles
// ---------------------------------------------------------
float L(vec2 uv, vec2 ofs, float b, float l) {
    return smoothstep(0.0, 800.0, b * max(0.1, l) / pow(max(0.000000001, length(uv - ofs)), 1.0 / max(0.1, l)));
}

float rand(vec2 co, float s) {
    float PHI = 1.61803398874989484820459;
    return fract(tan(distance(co * PHI, co) * s) * co.x);
}

vec2 H12(float s) {
    float x = rand(vec2(243.234, 63.834), s) - 0.5;
    float y = rand(vec2(53.1434, 13.1234), s) - 0.5;
    return vec2(x, y);
}

// ---------------------------------------------------------
// Fonctions pour les nuages
// ---------------------------------------------------------
float hash11(float p) {
    vec3 p3 = fract(vec3(p) * HASHSCALE1);
    p3 += dot(p3, p3.yzx + 19.19);
    return fract((p3.x + p3.y) * p3.z);
}

float hash12(vec2 p) {
    vec3 p3 = fract(vec3(p.xyx) * HASHSCALE1);
    p3 += dot(p3, p3.yzx + 19.19);
    return fract((p3.x + p3.y) * p3.z);
}

float smoothNoise13(in vec3 x) {
    vec3 p = floor(x);
    vec3 f = smoothstep(0.0, 1.0, fract(x));
    float n = p.x + p.y * 57.0 + 113.0 * p.z;
    return mix(
            mix(
                mix(hash11(n + 0.0),  hash11(n + 1.0),  f.x),
                mix(hash11(n + 57.0), hash11(n + 58.0), f.x),
                f.y),
            mix(
                mix(hash11(n + 113.0), hash11(n + 114.0), f.x),
                mix(hash11(n + 170.0), hash11(n + 171.0), f.x),
                f.y),
            f.z);
}

mat3 m = mat3(
    0.00,  1.60,  1.20,
   -1.60,  0.8,  -0.96,
   -1.20, -0.96,  1.28
);

float FractionalBrownianMotion(vec3 p) {
    float f = abs(sin(uTime * 0.1) + 0.3) * 0.5000 * smoothNoise13(p);
    p = m * p * 1.2;
    f += -0.52500 * smoothNoise13(p);
    p = m * p * 1.3;
    f += 0.1666 * smoothNoise13(p);
    p = m * p * 1.4;
    f += 0.0834 * smoothNoise13(p);
    return f;
}

// ---------------------------------------------------------
// Fonction principale (inspirée de mainImage de Shadertoy)
// ---------------------------------------------------------
void main(void) {
    vec2 fragCoord = vUv0 * uResolution;

    // 1. Préparation des coordonnées pour le ciel étoilé
    vec2 uv = fragCoord / uResolution.xy;
    uv -= 0.5;
    uv.x *= (uResolution.x / uResolution.y);
    // On ajoute le décalage issu du mouvement de la caméra
    uv += uCamOffset;
    // Initialisation du fond
    vec4 col = vec4(0.0);
    
    // Fond dynamique coloré
    vec4 bColor = vec4(0.01176470588 * sin(uTime), 0.1098039215, 0.18117647058, 1.0);
    vec4 pColor = vec4(0.23333333333 * (uv.x * sin(uTime * 0.1) + uv.y * sin(uTime * 0.1)), 0.10843137254, 0.13725490196, 1.0);
    vec4 lbColor = vec4(0.69196078431 * sin(uTime * 0.21) * uv.x, 0.21568627451, 0.13333333333, 1.0);
    vec4 blb = mix(bColor, lbColor, -uv.x * 0.2 - (uv.y * 0.5));
    col += mix(blb, pColor, uv.x - (uv.y * 1.5));
    
    // 2. Génération du ciel étoilé
    for (float i = 0.0; i < NS; i++) {
        vec2 ofs = H12(i + 1.0);
        ofs *= vec2(1.8, 1.1);
        float r = (mod(i, 20.0) == 0.0) ? 0.25 + abs(sin(i / 50.0)) : 0.25;
        float intensity = r + (sin(fract(uTime) * 0.1 * i) + 1.0) * 0.02;
        col += vec4(L(uv, ofs, intensity, 1.0));
    }
    
    // 3. Superposition des nuages sur le ciel étoilé
    vec2 normUV = fragCoord.xy / uResolution.xy;
    float timeVal = uTime * 0.05;
    vec3 vFbmInput = vec3(normUV.x - timeVal, normUV.y, 0.0);
    vec3 vFogColor = vec3(0.7, 0.7, 0.9);
    col += normUV.y * vec4(vFogColor * FractionalBrownianMotion(vFbmInput), 0.0);
    
    // Lecture de la scène déjà rendue
    vec4 sceneCol = texture2D(uColorBuffer, vUv0);
    
    // Lecture de la profondeur pour déterminer les zones "vides"
    float depthVal = getLinearScreenDepth(vUv0);
    float farPlane = camera_params.y;
    float threshold = 0.99;
    float isBackground = step(farPlane * threshold, depthVal);
    
    // Mix final : dans les zones sans géométrie (isBackground==1), on superpose notre effet,
    // modulé par uFade pour permettre un fondu progressif.
    vec4 finalColor = mix(sceneCol, col, isBackground * uFade);
    gl_FragColor = finalColor;
}
`;

    this.shader = pc.createShaderFromCode(
        graphicsDevice,
        pc.PostEffect.quadVertexShader,
        fshader,
        "StarfieldDepthShader",
        { aPosition: pc.SEMANTIC_POSITION }
    );
}
StarfieldDepthEffect.prototype = Object.create(pc.PostEffect.prototype);
StarfieldDepthEffect.prototype.constructor = StarfieldDepthEffect;

Object.assign(StarfieldDepthEffect.prototype, {
    render: function (inputTarget, outputTarget, rect) {
        var device = this.device;
        var scope = device.scope;
        
        scope.resolve("uColorBuffer").setValue(inputTarget.colorBuffer);
        scope.resolve("uResolution").setValue([device.width, device.height]);
        scope.resolve("uTime").setValue(this.time);
        scope.resolve("uFade").setValue(this.fade);
        scope.resolve("uCamOffset").setValue([this.camOffset.x, this.camOffset.y]);
        
        this.drawQuad(outputTarget, this.shader, rect);
    }
});
