// ribbon.js
var Ribbon = pc.createScript('ribbon');

Ribbon.attributes.add("lifetime", { type: "number", default: 0.5 });
Ribbon.attributes.add("height",   { type: "number", default: 0.4 });
Ribbon.attributes.add("size",     { type: "number", default: 1.0 });

var MAX_VERTICES = 600;
var VERTEX_SIZE  = 4;

Ribbon.prototype.initialize = function () {
    const vshader = `
    attribute vec4 aPositionAge;
    uniform mat4 matrix_viewProjection;
    uniform float trail_time;

    varying float vAge;
    varying vec3  vPos;

    void main(void) {
        vAge = trail_time - aPositionAge.w;
        vPos = aPositionAge.xyz;
        gl_Position = matrix_viewProjection * vec4(aPositionAge.xyz, 1.0);
    }
    `;

    const fshader = `
    precision mediump float;

    uniform float trail_lifetime;

    varying float vAge;
    varying vec3  vPos;

    void main(void) {
        float t = vAge / trail_lifetime;
        float alpha = 1.0 - t;
        if (alpha <= 0.0) discard;

        float stripesCount = 25.0;
        float uv = (vPos.x + vPos.y) + (t * stripesCount);

        float bandIndex = floor(uv);
        vec3 color;
        if (mod(bandIndex, 2.0) < 0.5) {
            color = vec3(1.0, 1.0, 1.0);
        } else {
            color = vec3(1.0, 0.0, 1.0);
        }
        gl_FragColor = vec4(color, alpha);
    }
    `;

    var shaderDefinition = {
        attributes: { aPositionAge: pc.SEMANTIC_POSITION },
        vshader: vshader,
        fshader: fshader
    };

    this.material = new pc.Material();
    this.material.shader = new pc.Shader(this.app.graphicsDevice, shaderDefinition);
    this.material.setParameter('trail_time', 0);
    this.material.setParameter('trail_lifetime', this.lifetime);
    this.material.cull = pc.CULLFACE_NONE;
    this.material.blend = true;
    this.material.blendSrc = pc.BLENDMODE_SRC_ALPHA;
    this.material.blendDst = pc.BLENDMODE_ONE_MINUS_SRC_ALPHA;
    this.material.blendEquation = pc.BLENDEQUATION_ADD;
    this.material.depthWrite = false;

    this.timer = 0;
    this.vertices = [];
    this.vertexData = new Float32Array(MAX_VERTICES * VERTEX_SIZE);
    this.vertexIndexArray = [];
    for (var i = 0; i < MAX_VERTICES; i++) this.vertexIndexArray.push(i);

    this.mesh = new pc.Mesh(this.app.graphicsDevice);
    this.mesh.clear(true, false);
    this.mesh.setPositions(this.vertexData, VERTEX_SIZE, MAX_VERTICES);
    this.mesh.setIndices(this.vertexIndexArray, MAX_VERTICES);
    this.mesh.update(pc.PRIMITIVE_TRISTRIP);

    var meshInstance = new pc.MeshInstance(this.mesh, this.material);
    this.entity.addComponent('render', {
        meshInstances: [meshInstance],
        layers: [this.app.scene.layers.getLayerByName('World').id]
    });

    this.entity.render.enabled = false;
};

Ribbon.prototype.reset = function () {
    this.timer = 0;
    this.vertices = [];
};

Ribbon.prototype.spawnNewVertices = function () {
    var pos = this.entity.getPosition();
    pos.x += Math.sin(this.timer * 2) * 0.3 * this.size;
    pos.y += Math.cos(this.timer * 2) * 0.2 * this.size;
    pos.z += Math.sin(this.timer * 3) * 0.3 * this.size;

    var yaxis = this.entity.up.clone().scale(this.height * this.size);
    var spawnTime = this.timer;
    var vertexPair = [pos.x, pos.y, pos.z, pos.x, pos.y, pos.z];

    this.vertices.unshift({
        spawnTime: spawnTime,
        vertexPair: vertexPair,
        yaxis: yaxis
    });
};

Ribbon.prototype.clearOldVertices = function () {
    for (var i = this.vertices.length - 1; i >= 0; i--) {
        if (this.timer - this.vertices[i].spawnTime >= this.lifetime) {
            this.vertices.pop();
        } else {
            break;
        }
    }
};

Ribbon.prototype.prepareVertexData = function () {
    for (var i = 0; i < this.vertices.length; i++) {
        var vp = this.vertices[i];
        var age = this.timer - vp.spawnTime;
        var frac = 1.0 - (age / this.lifetime);
        if (frac < 0) frac = 0;

        this.vertexData[i * 8 + 0] = vp.vertexPair[0] + vp.yaxis.x * frac;
        this.vertexData[i * 8 + 1] = vp.vertexPair[1] + vp.yaxis.y * frac;
        this.vertexData[i * 8 + 2] = vp.vertexPair[2] + vp.yaxis.z * frac;
        this.vertexData[i * 8 + 3] = vp.spawnTime;

        this.vertexData[i * 8 + 4] = vp.vertexPair[3] - vp.yaxis.x * frac;
        this.vertexData[i * 8 + 5] = vp.vertexPair[4] - vp.yaxis.y * frac;
        this.vertexData[i * 8 + 6] = vp.vertexPair[5] - vp.yaxis.z * frac;
        this.vertexData[i * 8 + 7] = vp.spawnTime;
    }
};

Ribbon.prototype.update = function (dt) {
    this.timer += dt;
    this.material.setParameter('trail_time', this.timer);
    this.clearOldVertices();
    this.spawnNewVertices();

    if (this.vertices.length > 1) {
        var currentLength = this.vertices.length * 2;
        var limit = Math.min(currentLength, MAX_VERTICES);
        this.prepareVertexData();
        this.mesh.setPositions(this.vertexData, VERTEX_SIZE, limit);
        this.mesh.setIndices(this.vertexIndexArray, limit);
        this.mesh.update(pc.PRIMITIVE_TRISTRIP);
        this.entity.render.enabled = true;
    }
};
