/* rotateOrbit.js
 * --------------------------------------------------------
 * Script PlayCanvas : fait orbiter une caméra autour d’une entité cible.
 * La caméra regarde toujours la cible.
 * --------------------------------------------------------
 * À ajouter sur une entité caméra (ou n’importe quelle entité « suivie »).
 */

var RotateOrbit = pc.createScript('rotateOrbit');

// === ATTRIBUTS EXPOSÉS DANS L’ÉDITEUR ===

// Entité cible (laisser vide → aucune orbite)
RotateOrbit.attributes.add('targetEntity', {
    type: 'entity',
    title: 'Target Entity',
    description: 'Entité à orbiter et regarder'
});

// Rayon de l’orbite
RotateOrbit.attributes.add('distance', {
    type: 'number',
    default: 5,
    title: 'Distance',
    description: 'Rayon (en unités PlayCanvas)'
});

// Vitesse de rotation (degrés par seconde)
RotateOrbit.attributes.add('speed', {
    type: 'number',
    default: 30,
    title: 'Rotation Speed (deg/s)'
});

// Axe de rotation (1 = actif, 0 = inactif) pour X, Y, Z
RotateOrbit.attributes.add('axis', {
    type: 'vec3',
    default: [0, 1, 0],
    title: 'Rotation Axis',
    description: 'Vecteur (1/0) définissant les axes actifs'
});

// Angles de départ (Euler – degrés)
RotateOrbit.attributes.add('initialAngles', {
    type: 'vec3',
    default: [0, 0, 0],
    title: 'Initial Angles',
    description: 'Orientation initiale de la caméra (degrés)'
});

// === INITIALISATION ===
RotateOrbit.prototype.initialize = function () {
    // Copie des angles en degrés
    this._angles = new pc.Vec3(
        this.initialAngles.x,
        this.initialAngles.y,
        this.initialAngles.z
    );

    // Vecteur pointant vers l’arrière (local Z) à transformer
    this._baseOffset = new pc.Vec3(0, 0, this.distance * -1);
};

// === MISE À JOUR À CHAQUE FRAME ===
RotateOrbit.prototype.update = function (dt) {
    var target = this.targetEntity;
    if (!target) return; // Rien à faire sans cible

    // Avancer les angles selon la vitesse et l’axe actif
    this._angles.x += this.axis.x * this.speed * dt;
    this._angles.y += this.axis.y * this.speed * dt;
    this._angles.z += this.axis.z * this.speed * dt;

    // Construire la rotation à partir des angles actuels (degrés -> quat)
    var rot = new pc.Quat();
    rot.setFromEulerAngles(this._angles.x, this._angles.y, this._angles.z);

    // Calculer la position orbitale
    var offset = new pc.Vec3();
    rot.transformVector(this._baseOffset, offset);

    var targetPos = target.getPosition();
    var camPos = new pc.Vec3();
    camPos.add2(targetPos, offset);

    this.entity.setPosition(camPos);
    this.entity.lookAt(targetPos);
};

// === HOT‑RELOAD (PlayCanvas Editor) ===
RotateOrbit.prototype.swap = function (old) {
    // Préserver les angles durant le hot‑reload
    this._angles = old && old._angles ? old._angles.clone() : this._angles;
};
