#!/usr/bin/env python3
"""
DayDream Configuration Editor
A tabbed, menu-driven configuration editor for DayDream BBS
"""

import curses
import sys
import os
from typing import List, Optional, Tuple, Dict, Any
from config_parser import DayDreamConfigParser, SECTION_FIELDS, FIELD_CHOICES, ConfigEntry


class Colors:
    """Color pair constants for the interface"""
    NORMAL = 1
    HEADER = 2
    SELECTED = 3
    INACTIVE = 4
    ERROR = 5
    SUCCESS = 6
    FIELD_LABEL = 7
    FIELD_VALUE = 8
    TAB_ACTIVE = 9
    TAB_INACTIVE = 10


class ConfigEditor:
    """Main configuration editor class"""
    
    def __init__(self, config_file: str):
        self.config_file = config_file
        self.parser = DayDreamConfigParser()
        self.current_tab = 0
        self.current_selection = 0
        self.scroll_offset = 0
        self.in_edit_mode = False
        self.status_message = ""
        self.status_type = "info"
        self.stdscr = None
        self.expanded_items = {}  # Track which items are expanded
        self.tab_scroll_offset = 0  # Track tab scrolling
        
        # Tab definitions
        self.tabs = [
            ("System", "DAYDREAM.DAT"),
            ("Conferences", "CONFERENCES.DAT"),
            ("Doors", "EXTERNALCOMMANDS.DAT"),
            ("Access", "ACCESS.DAT"),
            ("Archivers", "ARCHIVERS.DAT"),
            ("Display", "DISPLAY.DAT"),
            ("Nodes", "MULTINODE.DAT"),
            ("Security", "SECURITY.DAT"),
            ("Protocols", "PROTOCOLS.DAT"),
        ]
        
        # Load configuration
        try:
            self.parser.parse_file(config_file)
            self.status_message = f"Loaded {config_file}"
            self.status_type = "success"
        except Exception as e:
            self.status_message = f"Error loading {config_file}: {str(e)}"
            self.status_type = "error"
            
    def init_colors(self):
        """Initialize color pairs"""
        curses.start_color()
        curses.use_default_colors()
        
        # Define color pairs
        curses.init_pair(Colors.NORMAL, curses.COLOR_WHITE, -1)
        curses.init_pair(Colors.HEADER, curses.COLOR_CYAN, -1)
        curses.init_pair(Colors.SELECTED, curses.COLOR_BLACK, curses.COLOR_WHITE)
        curses.init_pair(Colors.INACTIVE, curses.COLOR_BLACK, curses.COLOR_BLUE)
        curses.init_pair(Colors.ERROR, curses.COLOR_RED, -1)
        curses.init_pair(Colors.SUCCESS, curses.COLOR_GREEN, -1)
        curses.init_pair(Colors.FIELD_LABEL, curses.COLOR_YELLOW, -1)
        curses.init_pair(Colors.FIELD_VALUE, curses.COLOR_WHITE, -1)
        curses.init_pair(Colors.TAB_ACTIVE, curses.COLOR_BLACK, curses.COLOR_CYAN)
        curses.init_pair(Colors.TAB_INACTIVE, curses.COLOR_WHITE, curses.COLOR_BLUE)
        
    def draw_tabs(self, stdscr):
        """Draw the tab bar at the top with scrolling support"""
        height, width = stdscr.getmaxyx()
        
        # Clear tab line
        stdscr.hline(0, 0, ' ', width)
        
        # Calculate which tabs to show
        self.update_tab_scroll(width)
        
        x = 0
        visible_tabs = []
        
        # Show scroll indicator if needed
        if self.tab_scroll_offset > 0:
            try:
                stdscr.addstr(0, 0, "<", curses.color_pair(Colors.HEADER))
                x = 2
            except curses.error:
                pass
        
        # Draw visible tabs
        for i in range(self.tab_scroll_offset, len(self.tabs)):
            tab_name, _ = self.tabs[i]
            tab_text = f" {tab_name} "
            
            # Check if this tab fits
            needed_width = len(tab_text) + 1
            if i < len(self.tabs) - 1:  # Not the last tab
                needed_width += 3  # Space for ">" indicator
                
            if x + needed_width > width:
                # Show scroll indicator
                try:
                    stdscr.addstr(0, width - 1, ">", curses.color_pair(Colors.HEADER))
                except curses.error:
                    pass
                break
                
            if i == self.current_tab:
                attr = curses.color_pair(Colors.TAB_ACTIVE) | curses.A_BOLD
            else:
                attr = curses.color_pair(Colors.TAB_INACTIVE)
                
            try:
                stdscr.addstr(0, x, tab_text, attr)
            except curses.error:
                pass
                
            visible_tabs.append(i)
            x += len(tab_text) + 1
            
    def update_tab_scroll(self, screen_width):
        """Update tab scrolling to ensure current tab is visible"""
        # Calculate tab widths
        tab_widths = []
        total_width = 0
        
        for tab_name, _ in self.tabs:
            width = len(tab_name) + 3  # " name " + separator
            tab_widths.append(width)
            total_width += width
            
        # If all tabs fit, no scrolling needed
        if total_width <= screen_width - 4:  # Leave space for scroll indicators
            self.tab_scroll_offset = 0
            return
            
        # Ensure current tab is visible
        current_tab_start = sum(tab_widths[:self.current_tab])
        current_tab_end = current_tab_start + tab_widths[self.current_tab]
        
        # Calculate visible range based on current scroll
        visible_start = sum(tab_widths[:self.tab_scroll_offset])
        visible_end = visible_start
        
        # Calculate how many tabs fit in the visible area
        available_width = screen_width - 4  # Leave space for scroll indicators
        x = 0
        
        for i in range(self.tab_scroll_offset, len(self.tabs)):
            if x + tab_widths[i] > available_width:
                break
            x += tab_widths[i]
            visible_end += tab_widths[i]
            
        # Adjust scroll if current tab is not visible
        if self.current_tab < self.tab_scroll_offset:
            # Current tab is to the left, scroll left
            self.tab_scroll_offset = self.current_tab
        elif current_tab_end > visible_end:
            # Current tab is to the right, scroll right
            # Find the rightmost starting position that shows the current tab
            for start_tab in range(self.current_tab, -1, -1):
                test_width = 0
                for i in range(start_tab, len(self.tabs)):
                    if test_width + tab_widths[i] > available_width:
                        break
                    test_width += tab_widths[i]
                    if i == self.current_tab:
                        self.tab_scroll_offset = start_tab
                        return
            
    def draw_status(self, stdscr):
        """Draw status line at bottom"""
        height, width = stdscr.getmaxyx()
        status_y = height - 1
        
        # Clear status line
        stdscr.hline(status_y, 0, ' ', width)
        
        # Status message
        if self.status_message:
            if self.status_type == "error":
                attr = curses.color_pair(Colors.ERROR)
            elif self.status_type == "success":
                attr = curses.color_pair(Colors.SUCCESS)
            else:
                attr = curses.color_pair(Colors.NORMAL)
                
            try:
                stdscr.addstr(status_y, 2, self.status_message[:width-4], attr)
            except curses.error:
                pass
                
        # Help text
        help_text = "TAB/←→:Switch F2:Save F3:Add F4:Del F10:Quit"
        try:
            stdscr.addstr(status_y, width - len(help_text) - 2, help_text, 
                         curses.color_pair(Colors.HEADER))
        except curses.error:
            pass
            
    def get_current_section_data(self) -> List[Dict[str, Any]]:
        """Get flattened data for the current tab's section for display"""
        _, section_name = self.tabs[self.current_tab]
        section = self.parser.get_section(section_name)
        
        if not section:
            return []
            
        data = []
        
        if section_name == "CONFERENCES.DAT":
            # Special handling for conferences
            for i, subsection in enumerate(section.subsections):
                conf_name = "Unknown Conference"
                conf_num = str(i+1)
                
                # Get conference info
                conf_entries = []
                msg_base_entries = []
                current_msg_base = []
                
                for entry in subsection.entries:
                    if entry.key == "CONF_NAME":
                        conf_name = entry.value
                    elif entry.key == "CONF_NUMB":
                        conf_num = entry.value
                    
                    # Separate conference settings from message base settings
                    if entry.key.startswith("BASE_"):
                        if entry.key == "BASE_NUMBER" and current_msg_base:
                            # Start of new message base
                            msg_base_entries.append(current_msg_base)
                            current_msg_base = [entry]
                        else:
                            current_msg_base.append(entry)
                    else:
                        conf_entries.append(entry)
                
                # Add the last message base
                if current_msg_base:
                    msg_base_entries.append(current_msg_base)
                
                # Add conference header
                conf_key = f"{section_name}_conf_{i}"
                conf_expanded = self.expanded_items.get(conf_key, False)
                
                data.append({
                    'type': 'conference',
                    'name': f"Conference {conf_num}: {conf_name}",
                    'index': i,
                    'key': conf_key,
                    'expanded': conf_expanded,
                    'entries': conf_entries,
                    'level': 0
                })
                
                # Add conference entries if expanded
                if conf_expanded:
                    for entry in conf_entries:
                        field_desc = entry.key
                        fields = SECTION_FIELDS[section_name].get('conference', [])
                        for field_key, desc, _ in fields:
                            if field_key == entry.key:
                                field_desc = desc
                                break
                        
                        data.append({
                            'type': 'conf_entry',
                            'name': f"{field_desc}: {entry.value}",
                            'entry': entry,
                            'field_info': None,
                            'level': 1
                        })
                    
                    # Add message bases
                    for j, msg_base in enumerate(msg_base_entries):
                        base_name = "Unknown Base"
                        base_num = str(j+1)
                        
                        for entry in msg_base:
                            if entry.key == "BASE_NAME":
                                base_name = entry.value
                            elif entry.key == "BASE_NUMBER":
                                base_num = entry.value
                        
                        base_key = f"{conf_key}_base_{j}"
                        base_expanded = self.expanded_items.get(base_key, False)
                        
                        data.append({
                            'type': 'messagebase',
                            'name': f"Base {base_num}: {base_name}",
                            'index': j,
                            'key': base_key,
                            'expanded': base_expanded,
                            'entries': msg_base,
                            'level': 1
                        })
                        
                        # Add message base entries if expanded
                        if base_expanded:
                            for entry in msg_base:
                                field_desc = entry.key
                                fields = SECTION_FIELDS[section_name].get('messagebase', [])
                                for field_key, desc, _ in fields:
                                    if field_key == entry.key:
                                        field_desc = desc
                                        break
                                
                                data.append({
                                    'type': 'base_entry',
                                    'name': f"{field_desc}: {entry.value}",
                                    'entry': entry,
                                    'field_info': None,
                                    'level': 2
                                })
                
        elif section_name in ["EXTERNALCOMMANDS.DAT", "ACCESS.DAT", "ARCHIVERS.DAT", 
                             "DISPLAY.DAT", "MULTINODE.DAT", "SECURITY.DAT", "PROTOCOLS.DAT"]:
            # Sections with subsections
            for i, subsection in enumerate(section.subsections):
                # Get a descriptive name
                name_key = None
                if section_name == "EXTERNALCOMMANDS.DAT":
                    name_key = "DOOR_COMMAND"
                elif section_name == "ACCESS.DAT":
                    name_key = "PRESET_DESC"
                elif section_name == "ARCHIVERS.DAT":
                    name_key = "ARC_NAME"
                elif section_name == "DISPLAY.DAT":
                    name_key = "DPL_PATH"
                elif section_name == "MULTINODE.DAT":
                    name_key = "MNODE_NODE"
                elif section_name == "SECURITY.DAT":
                    name_key = "SEC_LEVEL"
                elif section_name == "PROTOCOLS.DAT":
                    name_key = "XPR_NAME"
                    
                item_name = f"Entry {i+1}"
                if name_key:
                    for entry in subsection.entries:
                        if entry.key == name_key:
                            item_name = f"{entry.value}"
                            break
                            
                item_key = f"{section_name}_item_{i}"
                item_expanded = self.expanded_items.get(item_key, False)
                
                data.append({
                    'type': 'subsection',
                    'name': item_name,
                    'index': i,
                    'key': item_key,
                    'expanded': item_expanded,
                    'entries': subsection.entries,
                    'level': 0
                })
                
                # Add subsection entries if expanded
                if item_expanded:
                    for entry in subsection.entries:
                        field_desc = entry.key
                        fields = SECTION_FIELDS.get(section_name, [])
                        for field_key, desc, _ in fields:
                            if field_key == entry.key:
                                field_desc = desc
                                break
                        
                        data.append({
                            'type': 'sub_entry',
                            'name': f"{field_desc}: {entry.value}",
                            'entry': entry,
                            'field_info': None,
                            'level': 1
                        })
        else:
            # Simple section with direct entries
            for entry in section.entries:
                field_info = None
                fields = SECTION_FIELDS.get(section_name, [])
                for field_key, field_desc, field_type in fields:
                    if field_key == entry.key:
                        field_info = (field_desc, field_type)
                        break
                        
                data.append({
                    'type': 'entry',
                    'name': entry.key,
                    'value': entry.value,
                    'field_info': field_info,
                    'entry': entry,
                    'level': 0
                })
                
        return data
        
    def draw_section_content(self, stdscr, start_y: int, max_height: int):
        """Draw the content for the current section"""
        data = self.get_current_section_data()
        
        if not data:
            try:
                stdscr.addstr(start_y + 2, 4, "No data available", 
                             curses.color_pair(Colors.INACTIVE))
            except curses.error:
                pass
            return
            
        height, width = stdscr.getmaxyx()
        display_width = width - 8
        
        y = start_y
        
        for i, item in enumerate(data):
            if i < self.scroll_offset:
                continue
                
            if y >= start_y + max_height:
                break
                
            # Highlight current selection
            if i == self.current_selection:
                attr = curses.color_pair(Colors.SELECTED)
            else:
                attr = curses.color_pair(Colors.NORMAL)
            
            # Calculate indentation based on level
            indent = "  " * item.get('level', 0)
            
            try:
                if item['type'] in ['conference', 'subsection', 'messagebase']:
                    # Collapsible item
                    prefix = "[-] " if item.get('expanded') else "[+] "
                    text = f"{indent}{prefix}{item['name']}"
                    stdscr.addstr(y, 4, text[:display_width], attr)
                    y += 1
                    
                elif item['type'] in ['entry', 'conf_entry', 'sub_entry', 'base_entry']:
                    # Entry item
                    if item['type'] == 'entry':
                        # Direct entry in simple sections
                        if item['field_info']:
                            desc, _ = item['field_info']
                            text = f"{indent}{desc}: {item['value']}"
                        else:
                            text = f"{indent}{item['name']}: {item['value']}"
                    else:
                        # Sub-entry
                        text = f"{indent}{item['name']}"
                        
                    stdscr.addstr(y, 4, text[:display_width], attr)
                    y += 1
                    
            except curses.error:
                pass
            
    def handle_key(self, key: int) -> bool:
        """Handle keyboard input. Returns False to quit."""
        data = self.get_current_section_data()
        
        if key == ord('\t') or key == curses.KEY_RIGHT:
            # Switch tabs
            self.current_tab = (self.current_tab + 1) % len(self.tabs)
            self.current_selection = 0
            self.scroll_offset = 0
            
        elif key == curses.KEY_LEFT:
            # Switch tabs backwards
            self.current_tab = (self.current_tab - 1) % len(self.tabs)
            self.current_selection = 0
            self.scroll_offset = 0
            
        elif key == curses.KEY_UP:
            if self.current_selection > 0:
                self.current_selection -= 1
                if self.current_selection < self.scroll_offset:
                    self.scroll_offset = self.current_selection
                    
        elif key == curses.KEY_DOWN:
            if self.current_selection < len(data) - 1:
                self.current_selection += 1
                # Adjust scroll if needed
                height = curses.LINES - 6  # Account for tabs and status
                if self.current_selection >= self.scroll_offset + height:
                    self.scroll_offset = self.current_selection - height + 1
                    
        elif key == ord('\n') or key == curses.KEY_ENTER or key == 10:
            # Toggle expansion or edit entry
            if data and self.current_selection < len(data):
                item = data[self.current_selection]
                if item['type'] in ['conference', 'subsection', 'messagebase']:
                    # Toggle expansion state
                    item_key = item.get('key')
                    if item_key:
                        current_state = self.expanded_items.get(item_key, False)
                        self.expanded_items[item_key] = not current_state
                elif item['type'] in ['entry', 'conf_entry', 'sub_entry', 'base_entry']:
                    self.edit_entry(item)
                    
        elif key == curses.KEY_F2:
            # Save
            self.save_config()
            
        elif key == curses.KEY_F3:
            # Add new entry
            self.add_entry()
            
        elif key == curses.KEY_F4:
            # Delete entry
            self.delete_entry()
            
        elif key == curses.KEY_F10 or key == 27:  # F10 or ESC
            return False
            
        return True
        
    def edit_entry(self, item: Dict[str, Any]):
        """Edit a configuration entry"""
        if item['type'] not in ['entry', 'conf_entry', 'sub_entry', 'base_entry']:
            return
            
        if 'entry' not in item:
            self.status_message = "Cannot edit this item"
            self.status_type = "error"
            return
            
        entry = item['entry']
        field_info = item.get('field_info')
        
        # Get field info based on entry type and section
        if not field_info:
            _, section_name = self.tabs[self.current_tab]
            fields = []
            
            if item['type'] == 'entry':
                fields = SECTION_FIELDS.get(section_name, [])
            elif item['type'] == 'conf_entry':
                fields = SECTION_FIELDS.get(section_name, {}).get('conference', [])
            elif item['type'] == 'base_entry':
                fields = SECTION_FIELDS.get(section_name, {}).get('messagebase', [])
            elif item['type'] == 'sub_entry':
                fields = SECTION_FIELDS.get(section_name, [])
            
            for field_key, field_desc, field_type in fields:
                if field_key == entry.key:
                    field_info = (field_desc, field_type)
                    break
        
        if not field_info:
            # Simple text edit
            new_value = self.get_string_input(self.stdscr, f"Edit {entry.key}", entry.value)
            if new_value is not None and new_value != entry.value:
                entry.value = new_value
                self.status_message = f"Updated {entry.key}"
                self.status_type = "success"
        else:
            desc, field_type = field_info
            
            if field_type == 'yesno':
                # Toggle Y/N
                old_value = entry.value
                entry.value = 'N' if entry.value == 'Y' else 'Y'
                self.status_message = f"Toggled {desc}: {old_value} -> {entry.value}"
                self.status_type = "success"
            elif field_type == 'choice':
                # Show choices
                choices = FIELD_CHOICES.get(entry.key, [])
                if choices:
                    choice = self.show_choice_menu(self.stdscr, desc, choices, entry.value)
                    if choice is not None and choice != entry.value:
                        old_value = entry.value
                        entry.value = choice
                        self.status_message = f"Updated {desc}: {old_value} -> {choice}"
                        self.status_type = "success"
                else:
                    new_value = self.get_string_input(self.stdscr, f"Edit {desc}", entry.value)
                    if new_value is not None and new_value != entry.value:
                        entry.value = new_value
                        self.status_message = f"Updated {desc}"
                        self.status_type = "success"
            else:
                # Text/number/path/password edit
                new_value = self.get_string_input(self.stdscr, f"Edit {desc}", entry.value)
                if new_value is not None and new_value != entry.value:
                    entry.value = new_value
                    self.status_message = f"Updated {desc}"
                    self.status_type = "success"
                    
    def get_string_input(self, stdscr, prompt: str, initial_value: str = "") -> Optional[str]:
        """Get string input from user"""
        # Set up for input
        curses.echo()
        curses.curs_set(1)
        stdscr.nodelay(False)  # Blocking input
        
        try:
            height, width = stdscr.getmaxyx()
            y = height // 2 - 2
            x = 2
            
            # Clear area (larger area)
            for i in range(6):
                try:
                    stdscr.hline(y + i, 0, ' ', width)
                except:
                    pass
            
            # Draw input box
            try:
                stdscr.addstr(y, x, f"Edit: {prompt}", curses.color_pair(Colors.HEADER))
                stdscr.addstr(y + 1, x, f"Current: {initial_value}")
                stdscr.addstr(y + 2, x, f"New: ")
                stdscr.addstr(y + 4, x, "Enter new value (or press Enter to keep current):")
                stdscr.addstr(y + 5, x, "Press Ctrl+C to cancel")
                stdscr.refresh()
                
                # Position cursor and get input
                stdscr.move(y + 2, x + 5)
                result = stdscr.getstr(y + 2, x + 5, min(60, width - x - 6)).decode('utf-8', errors='ignore')
                
                # If empty, keep original value
                if not result.strip():
                    return initial_value
                    
                return result
                
            except KeyboardInterrupt:
                return None
            except curses.error:
                return None
                
        except Exception:
            return None
            
        finally:
            # Restore settings
            curses.noecho()
            curses.curs_set(0)
            stdscr.nodelay(True)  # Non-blocking for main loop
            
    def show_choice_menu(self, stdscr, prompt: str, choices: List[Tuple[str, str]], 
                        current: str) -> Optional[str]:
        """Show a choice menu"""
        # Set up for input
        stdscr.nodelay(False)  # Blocking input
        
        try:
            # Find current selection
            current_choice = 0
            for i, (value, _) in enumerate(choices):
                if value == current:
                    current_choice = i
                    break
                    
            while True:
                height, width = stdscr.getmaxyx()
                menu_height = len(choices) + 4
                y = max(2, (height - menu_height) // 2)
                x = max(2, (width - 40) // 2)
                
                # Clear area
                for i in range(menu_height):
                    try:
                        stdscr.hline(y + i, 0, ' ', width)
                    except:
                        pass
                        
                try:
                    # Draw menu box
                    stdscr.addstr(y, x, f"Select {prompt}:", curses.color_pair(Colors.HEADER))
                    stdscr.addstr(y + 1, x, "-" * 35)
                    
                    for i, (value, desc) in enumerate(choices):
                        if i == current_choice:
                            attr = curses.color_pair(Colors.SELECTED) | curses.A_BOLD
                            prefix = "> "
                        else:
                            attr = curses.color_pair(Colors.NORMAL)
                            prefix = "  "
                        
                        text = f"{prefix}{value} - {desc}"
                        stdscr.addstr(y + 2 + i, x, text[:35], attr)
                        
                    stdscr.addstr(y + len(choices) + 2, x, "-" * 35)
                    stdscr.addstr(y + len(choices) + 3, x, "Up/Down: Navigate, Enter: Select, ESC: Cancel")
                    stdscr.refresh()
                    
                    key = stdscr.getch()
                    
                    if key == curses.KEY_UP and current_choice > 0:
                        current_choice -= 1
                    elif key == curses.KEY_DOWN and current_choice < len(choices) - 1:
                        current_choice += 1
                    elif key == ord('\n') or key == curses.KEY_ENTER or key == 10:
                        return choices[current_choice][0]
                    elif key == 27:  # ESC
                        return None
                        
                except curses.error:
                    return None
                    
        finally:
            stdscr.nodelay(True)  # Non-blocking for main loop
                
    def save_config(self):
        """Save configuration to file"""
        try:
            # Create backup
            backup_file = f"{self.config_file}.bak"
            if os.path.exists(self.config_file):
                import shutil
                shutil.copy2(self.config_file, backup_file)
                
            self.parser.write_file(self.config_file)
            self.status_message = f"Saved to {self.config_file}"
            self.status_type = "success"
        except Exception as e:
            self.status_message = f"Save failed: {str(e)}"
            self.status_type = "error"
            
    def add_entry(self):
        """Add a new entry to current section"""
        _, section_name = self.tabs[self.current_tab]
        data = self.get_current_section_data()
        
        if not data:
            self.status_message = "No data to use as template"
            self.status_type = "error"
            return
        
        # Determine what to add based on current selection and section
        if section_name == "CONFERENCES.DAT":
            self.add_conference_or_messagebase(data)
        elif section_name in ["EXTERNALCOMMANDS.DAT", "ACCESS.DAT", "ARCHIVERS.DAT", 
                             "DISPLAY.DAT", "MULTINODE.DAT", "SECURITY.DAT", "PROTOCOLS.DAT"]:
            self.add_subsection_entry(section_name, data)
        else:
            self.status_message = "Cannot add entries to this section"
            self.status_type = "error"
            
    def add_conference_or_messagebase(self, data):
        """Add either a conference or message base depending on context"""
        if not data:
            return
            
        # Find current selection context
        current_item = None
        if self.current_selection < len(data):
            current_item = data[self.current_selection]
        
        # Determine if we're adding a conference or message base
        choices = [
            ("conference", "New Conference"),
            ("messagebase", "New Message Base (to current conference)")
        ]
        
        choice = self.show_choice_menu(self.stdscr, "Add What?", choices, "conference")
        if not choice:
            return
            
        if choice == "conference":
            self.add_new_conference(data)
        elif choice == "messagebase":
            self.add_new_messagebase(data, current_item)
            
    def add_new_conference(self, data):
        """Add a new conference using the last conference as template"""
        section = self.parser.get_section("CONFERENCES.DAT")
        if not section or not section.subsections:
            self.status_message = "No existing conferences to use as template"
            self.status_type = "error"
            return
            
        # Use last conference as template
        template_conf = section.subsections[-1]
        
        # Create new conference subsection
        new_conf = self.parser.add_subsection("CONFERENCES.DAT")
        if not new_conf:
            self.status_message = "Failed to create conference"
            self.status_type = "error"
            return
            
        # Get next conference number
        next_conf_num = len(section.subsections)
        
        # Copy structure from template but with empty/default values
        conf_fields = SECTION_FIELDS["CONFERENCES.DAT"]["conference"]
        
        for field_key, field_desc, field_type in conf_fields:
            if field_key == "CONF_NUMB":
                new_conf.entries.append(ConfigEntry(key=field_key, value=str(next_conf_num)))
            elif field_key == "CONF_NAME":
                new_conf.entries.append(ConfigEntry(key=field_key, value=f"New Conference {next_conf_num}"))
            elif field_type == "yesno":
                new_conf.entries.append(ConfigEntry(key=field_key, value="N"))
            elif field_type == "number":
                new_conf.entries.append(ConfigEntry(key=field_key, value="0"))
            else:
                new_conf.entries.append(ConfigEntry(key=field_key, value="-"))
                
        self.status_message = f"Added Conference {next_conf_num}. Edit entries to configure."
        self.status_type = "success"
        
    def add_new_messagebase(self, data, current_item):
        """Add a new message base to the current conference"""
        if not current_item or current_item['type'] not in ['conference', 'conf_entry', 'messagebase', 'base_entry']:
            self.status_message = "Select a conference or message base first"
            self.status_type = "error"
            return
            
        # Find the conference this belongs to
        conf_index = None
        for i, item in enumerate(data):
            if item['type'] == 'conference':
                if i <= self.current_selection:
                    conf_index = item['index']
                else:
                    break
                    
        if conf_index is None:
            self.status_message = "Could not determine conference"
            self.status_type = "error"
            return
            
        section = self.parser.get_section("CONFERENCES.DAT")
        if not section or conf_index >= len(section.subsections):
            self.status_message = "Conference not found"
            self.status_type = "error"
            return
            
        conf_subsection = section.subsections[conf_index]
        
        # Count existing message bases in this conference
        base_count = 0
        for entry in conf_subsection.entries:
            if entry.key == "BASE_NUMBER":
                base_count += 1
                
        next_base_num = base_count + 1
        
        # Add message base entries
        base_fields = SECTION_FIELDS["CONFERENCES.DAT"]["messagebase"]
        
        for field_key, field_desc, field_type in base_fields:
            if field_key == "BASE_NUMBER":
                conf_subsection.entries.append(ConfigEntry(key=field_key, value=str(next_base_num)))
            elif field_key == "BASE_NAME":
                conf_subsection.entries.append(ConfigEntry(key=field_key, value=f"New Message Base {next_base_num}"))
            elif field_key == "BASE_TYPE":
                conf_subsection.entries.append(ConfigEntry(key=field_key, value="L"))
            elif field_key == "BASE_MAXMSGS":
                conf_subsection.entries.append(ConfigEntry(key=field_key, value="100"))
            elif field_type == "yesno":
                conf_subsection.entries.append(ConfigEntry(key=field_key, value="Y"))
            elif field_type == "choice":
                # Use first choice as default
                choices = FIELD_CHOICES.get(field_key, [])
                default_value = choices[0][0] if choices else "-"
                conf_subsection.entries.append(ConfigEntry(key=field_key, value=default_value))
            else:
                conf_subsection.entries.append(ConfigEntry(key=field_key, value="-"))
                
        self.status_message = f"Added Message Base {next_base_num} to conference. Edit entries to configure."
        self.status_type = "success"
        
    def add_subsection_entry(self, section_name, data):
        """Add a new subsection entry using the last entry as template"""
        section = self.parser.get_section(section_name)
        if not section or not section.subsections:
            self.status_message = "No existing entries to use as template"
            self.status_type = "error"
            return
            
        # Use last subsection as template
        template_subsection = section.subsections[-1]
        
        # Create new subsection
        new_subsection = self.parser.add_subsection(section_name)
        if not new_subsection:
            self.status_message = "Failed to create entry"
            self.status_type = "error"
            return
            
        # Get field definitions for this section
        fields = SECTION_FIELDS.get(section_name, [])
        
        # Copy structure from template but with empty/default values
        for field_key, field_desc, field_type in fields:
            if field_type == "yesno":
                new_subsection.entries.append(ConfigEntry(key=field_key, value="N"))
            elif field_type == "number":
                new_subsection.entries.append(ConfigEntry(key=field_key, value="0"))
            elif field_type == "choice":
                # Use first choice as default
                choices = FIELD_CHOICES.get(field_key, [])
                default_value = choices[0][0] if choices else "-"
                new_subsection.entries.append(ConfigEntry(key=field_key, value=default_value))
            else:
                new_subsection.entries.append(ConfigEntry(key=field_key, value="-"))
                
        self.status_message = f"Added new entry to {section_name}. Edit entries to configure."
        self.status_type = "success"
            
    def delete_entry(self):
        """Delete current entry"""
        data = self.get_current_section_data()
        if not data or self.current_selection >= len(data):
            return
            
        item = data[self.current_selection]
        _, section_name = self.tabs[self.current_tab]
        
        # Confirm deletion
        if item['type'] in ['conference', 'subsection']:
            item_desc = f"{item['type']} '{item['name']}'"
        elif item['type'] == 'messagebase':
            item_desc = f"message base '{item['name']}'"
        else:
            self.status_message = "Cannot delete this item"
            self.status_type = "error"
            return
            
        choices = [("Y", "Yes, delete it"), ("N", "No, cancel")]
        confirm = self.show_choice_menu(self.stdscr, f"Delete {item_desc}?", choices, "N")
        
        if confirm != "Y":
            self.status_message = "Delete cancelled"
            self.status_type = "info"
            return
        
        if item['type'] in ['conference', 'subsection']:
            # Delete entire conference or subsection
            if self.parser.delete_subsection(section_name, item['index']):
                self.status_message = f"Deleted {item_desc}"
                self.status_type = "success"
                if self.current_selection >= len(data) - 1:
                    self.current_selection = max(0, len(data) - 2)
            else:
                self.status_message = "Failed to delete entry"
                self.status_type = "error"
                
        elif item['type'] == 'messagebase':
            # Delete message base from conference
            self.delete_messagebase(data, item)
            
    def delete_messagebase(self, data, messagebase_item):
        """Delete a message base from its conference"""
        # Find the conference this message base belongs to
        conf_index = None
        for i, item in enumerate(data):
            if item['type'] == 'conference':
                if i <= self.current_selection:
                    conf_index = item['index']
                else:
                    break
                    
        if conf_index is None:
            self.status_message = "Could not find parent conference"
            self.status_type = "error"
            return
            
        section = self.parser.get_section("CONFERENCES.DAT")
        if not section or conf_index >= len(section.subsections):
            self.status_message = "Conference not found"
            self.status_type = "error"
            return
            
        conf_subsection = section.subsections[conf_index]
        
        # Find the message base entries to delete
        base_index = messagebase_item['index']
        entries_to_remove = []
        current_base_index = -1
        
        for i, entry in enumerate(conf_subsection.entries):
            if entry.key == "BASE_NUMBER":
                current_base_index += 1
                
            if current_base_index == base_index:
                entries_to_remove.append(i)
            elif current_base_index > base_index:
                break
                
        if not entries_to_remove:
            self.status_message = "Message base entries not found"
            self.status_type = "error"
            return
            
        # Remove entries in reverse order to maintain indices
        for i in reversed(entries_to_remove):
            del conf_subsection.entries[i]
            
        # Renumber remaining message bases
        self.renumber_messagebases(conf_subsection)
        
        self.status_message = f"Deleted message base from conference"
        self.status_type = "success"
        
        # Adjust selection
        if self.current_selection > 0:
            self.current_selection -= 1
            
    def renumber_messagebases(self, conf_subsection):
        """Renumber message bases after deletion to maintain sequential numbering"""
        base_number = 1
        
        for entry in conf_subsection.entries:
            if entry.key == "BASE_NUMBER":
                entry.value = str(base_number)
                base_number += 1
            
    def run(self, stdscr):
        """Main editor loop"""
        import time
        
        self.stdscr = stdscr
        self.init_colors()
        curses.curs_set(0)  # Hide cursor
        stdscr.nodelay(True)  # Non-blocking input
        
        needs_redraw = True
        
        while True:
            # Only redraw if needed
            if needs_redraw:
                stdscr.clear()
                height, width = stdscr.getmaxyx()
                
                # Draw interface
                self.draw_tabs(stdscr)
                self.draw_status(stdscr)
                
                # Draw section content
                content_start_y = 2
                content_height = height - 4
                self.draw_section_content(stdscr, content_start_y, content_height)
                
                stdscr.refresh()
                needs_redraw = False
            
            # Handle input
            try:
                key = stdscr.getch()
                if key != -1:  # Key was pressed
                    if not self.handle_key(key):
                        break
                    needs_redraw = True  # Redraw after key press
                else:
                    # No key pressed, sleep briefly to prevent busy loop
                    time.sleep(0.05)  # 50ms sleep
            except curses.error:
                time.sleep(0.05)


def main():
    """Main entry point"""
    if len(sys.argv) != 2:
        print("Usage: ddcfg_editor.py <config_file>")
        print("Example: ddcfg_editor.py daydream.cfg")
        sys.exit(1)
        
    config_file = sys.argv[1]
    
    if not os.path.exists(config_file):
        print(f"Error: Configuration file '{config_file}' not found")
        sys.exit(1)
        
    # Check terminal size
    try:
        import shutil
        cols, rows = shutil.get_terminal_size()
        if cols < 80 or rows < 25:
            print("Warning: Terminal should be at least 80x25 characters")
            print(f"Current size: {cols}x{rows}")
    except:
        pass
        
    editor = ConfigEditor(config_file)
    
    try:
        curses.wrapper(editor.run)
    except KeyboardInterrupt:
        print("\nEditor interrupted by user")
    except Exception as e:
        print(f"Editor error: {str(e)}")
        sys.exit(1)


if __name__ == "__main__":
    main()
